#pragma once

#include <auth/social/settings.h>
#include <common/http.h>
#include <common/context.h>
#include <common/account.h>
#include <common/json.h>
#include <common/errors.h>

#include <ymod_httpclient/cluster_client.h>
#include <ymod_ratecontroller/rate_controller.h>
#include <yplatform/util/execution_holder.h>
#include <yplatform/util/safe_call.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

#include <memory>
#include <string>

namespace xeno::auth::social {

using access_token_cb =
    std::function<void(error, const std::string& access_token, const time_point& expires_at)>;

struct get_access_token_op
{
    using yield_context = yplatform::yield_context<get_access_token_op>;

    get_access_token_op(
        context_ptr task_ctx,
        const std::string& server,
        const std::string& refresh_token,
        social_settings_ptr settings,
        const access_token_cb& cb)
        : task_ctx(task_ctx)
        , server(server)
        , refresh_token(refresh_token)
        , settings(settings)
        , cb(cb)
    {
        social_broker_client =
            yplatform::find<yhttp::cluster_client, std::shared_ptr>("social_broker_client");
        rate_controller =
            yplatform::find<ymod_ratecontroller::rate_controller_module>("rate_controller")
                ->get_controller("social." + server);
    }

    void operator()(yield_context yield_ctx, error err = {})
    {
        try
        {
            reenter(yield_ctx)
            {
                yield rate_controller->post(yield_ctx, task_ctx->uniq_id());
                if (err)
                {
                    YLOG_CTX_LOCAL(task_ctx, error)
                        << "getting access token error: " << err.message();
                    return yplatform::safe_call(task_ctx, "", cb, err, std::string(), time_point());
                }

                yield social_broker_client->async_run(task_ctx, make_request(), yield_ctx);

                if (err)
                {
                    YLOG_CTX_LOCAL(task_ctx, error)
                        << "getting access token error: " << err.message();
                    return yplatform::safe_call(task_ctx, "", cb, err, std::string(), time_point());
                }

                if (response.status != 200)
                {
                    YLOG_CTX_LOCAL(task_ctx, error)
                        << "getting access token error: bad status: " << response.status << " "
                        << response.reason;
                    return yplatform::safe_call(
                        task_ctx,
                        "",
                        cb,
                        code::get_access_token_error,
                        std::string(),
                        time_point());
                }

                err = extract_token();
                if (err)
                {
                    return yplatform::safe_call(task_ctx, "", cb, err, std::string(), time_point());
                }

                yplatform::safe_call(task_ctx, "", cb, error(), access_token, expires_at);
            }
        }
        catch (const std::exception& e)
        {
            YLOG_CTX_LOCAL(task_ctx, error)
                << "getting access token error: exception: " << e.what();
            yplatform::safe_call(
                task_ctx, "", cb, code::get_access_token_error, std::string(), time_point());
        }
    }

    void operator()(yield_context yield_ctx, error err, const yhttp::response& response)
    {
        if (!err)
        {
            this->response = response;
        }
        (*this)(yield_ctx, err);
    }

    void operator()(
        yield_context yield_ctx,
        error err,
        const ymod_ratecontroller::completion_handler& handler)
    {
        rc_holder = std::make_shared<yplatform::execution_holder>(handler);
        (*this)(yield_ctx, err);
    }

    yhttp::request make_request()
    {
        std::stringstream url, body;
        url << settings->access_url << server << "/refresh_token";
        if (settings->consumer.size())
        {
            url << "?consumer=" << settings->consumer;
        }
        body << "refresh_token=" << refresh_token;
        return http::request_t::POST(url.str(), body.str());
    }

    error extract_token()
    {
        json::value result;
        json::reader reader;
        if (!reader.parse(response.body, result))
        {
            YLOG_CTX_LOCAL(task_ctx, error) << "getting access token error: bad json";
            return code::get_access_token_error;
        }

        auto task = result["task"];
        auto state = task["state"];
        if (state.isNull() || state != "success")
        {
            auto reason = task["reason"];
            auto code = reason["code"].asString();
            auto description = reason["description"];
            YLOG_CTX_LOCAL(task_ctx, error)
                << "getting access token error: " << code
                << (description.isNull() ? "" : " " + reason["description"].asString());
            return (code == "invalid_token") ? code::invalid_token : code::get_access_token_error;
        }
        else
        {
            access_token = result["result"]["access_token"].asString();
            auto ttl = result["result"]["expires_in"].asUInt();
            auto penalty = time_traits::get_seconds_count(settings->access_token_ttl_penalty);
            if (ttl > penalty)
            {
                ttl -= penalty;
            }
            expires_at = time_traits::clock::now() + time_traits::seconds(ttl);
            return {};
        }
    }

    context_ptr task_ctx;
    std::string server;
    std::string refresh_token;
    social_settings_ptr settings;
    access_token_cb cb;
    std::shared_ptr<yhttp::cluster_client> social_broker_client;
    ymod_ratecontroller::rate_controller_ptr rate_controller;
    yplatform::execution_holder_ptr rc_holder;
    yhttp::response response;
    std::string access_token;
    time_point expires_at;
};

}

#include <yplatform/unyield.h>
