#pragma once

#include <auth/social/settings.h>
#include <common/context.h>
#include <common/errors.h>
#include <common/http.h>
#include <common/json.h>
#include <ymod_httpclient/cluster_client.h>
#include <chrono>

namespace xeno::auth::social {

namespace ph = std::placeholders;

struct refresh_token_response
{
    std::string error_message;
    std::string email;
    std::string oauth_app;
    std::string oauth_provider;
    std::string refresh_token;
    std::string access_token;
    time_point expires_at;
};
using refresh_token_cb = std::function<void(error, const refresh_token_response&)>;

inline void get_refresh_token(
    context_ptr ctx,
    http::request_t request,
    social_settings_ptr settings,
    const refresh_token_cb& cb)
{
    auto client = yplatform::find<yhttp::cluster_client, std::shared_ptr>("social_broker_client");
    client->async_run(
        ctx, std::move(request), [cb, settings, ctx](error err, yhttp::response response) {
            try
            {
                if (err)
                {
                    YLOG(ctx->logger(), error) << "getting refresh token error: " << err.message();
                    return cb(err, refresh_token_response());
                }
                if (response.status != 200)
                {
                    YLOG(ctx->logger(), error)
                        << "getting refresh token error: bad status: " << response.status << " "
                        << response.reason;
                    return cb(code::get_refresh_token_error, refresh_token_response());
                }
                json::value result;
                json::reader reader;
                if (!reader.parse(response.body, result))
                {
                    YLOG(ctx->logger(), error) << "getting refresh token error: bad json";
                    return cb(code::get_refresh_token_error, refresh_token_response());
                }
                auto error = result["error"];
                if (!error.isNull())
                {
                    auto description = result["description"];
                    YLOG(ctx->logger(), error)
                        << "getting refresh token error:"
                        << (description.isNull() ? "empty description" :
                                                   " " + description.asString());
                    return cb(code::get_refresh_token_error, refresh_token_response());
                }
                refresh_token_response response;
                response.email = result["profile"]["email"].asString();
                response.oauth_provider = result["profile"]["provider"]["name"].asString();
                auto token_node = result["token"];
                response.refresh_token = token_node["refresh"].asString();
                response.access_token = token_node["value"].asString();
                auto expire_ts = token_node["expires"].asInt64();
                auto cur_ts = std::chrono::duration_cast<std::chrono::seconds>(
                                  std::chrono::system_clock::now().time_since_epoch())
                                  .count();
                auto ttl = expire_ts > cur_ts ? expire_ts - cur_ts : 0;
                auto penalty = time_traits::get_seconds_count(settings->access_token_ttl_penalty);
                if (ttl > penalty)
                {
                    ttl -= penalty;
                }
                response.expires_at = time_traits::clock::now() + time_traits::seconds(ttl);
                response.oauth_app = token_node["application"].asString();
                cb({}, response);
            }
            catch (const std::exception& e)
            {
                YLOG(ctx->logger(), error)
                    << "getting refresh token error: exception: " << e.what();
                cb(code::operation_exception, refresh_token_response());
            }
        });
}

inline void get_refresh_token_by_task_id(
    context_ptr ctx,
    const std::string& task_id,
    social_settings_ptr settings,
    const refresh_token_cb& cb)
{
    std::stringstream url;
    url << settings->refresh_url << task_id;
    auto request = http::request_t::GET(url.str());
    get_refresh_token(ctx, std::move(request), settings, cb);
}

inline void get_refresh_token_by_ext_token(
    context_ptr ctx,
    const std::string& provider,
    const std::string& provider_token,
    const std::string& application,
    const std::string& scope,
    social_settings_ptr settings,
    const refresh_token_cb& cb)
{
    std::stringstream url, body;
    url << settings->task_by_token_url << "?provider=" << provider
        << "&application=" << application;
    if (settings->consumer.size())
    {
        url << "&consumer=" << settings->consumer;
    }
    body << "provider_token=" << provider_token << "&scope=" << scope;
    auto request = http::request_t::POST(url.str(), body.str());
    get_refresh_token(ctx, std::move(request), settings, cb);
}

}
