#include "errors.h"

namespace xeno {

struct error_category : public boost::system::error_category
{
    const char* name() const noexcept override
    {
        return "xeno::error_category";
    }

    std::string message(int v) const override
    {
        return error_message(static_cast<code>(v));
    }
};

struct error_condition_category : public boost::system::error_category
{
    const char* name() const noexcept override
    {
        return "xeno::error_condition_category";
    }

    std::string message(int v) const override
    {
        return errc_message(static_cast<errc>(v));
    }

    bool equivalent(const boost::system::error_code& ec, int condition) const noexcept override
    {
        errc e = static_cast<errc>(condition);
        switch (e)
        {
        case errc::ok:
            return !ec;

        case errc::perm_error:
            static const std::set<error> perm_error_codes = {
                code::user_from_yandex,       code::imap_protocol_disabled,
                code::imap_account_blocked,   code::folder_not_found,
                code::message_not_found,      code::label_not_found,
                code::threads_not_found,      code::parent_not_found,
                code::symbol_not_found,       code::need_parent_id,
                code::smtp_check_email_error, code::smtp_send_perm_error,
                code::body_is_empty,          code::bad_request,
                code::message_is_spam,        code::message_too_big,
                code::invalid_token,          code::bad_karma
            };
            return perm_error_codes.count(ec);

        case errc::imap_not_connected:
            static const std::set<error> not_connected_codes = { code::imap_connect_error,
                                                                 code::imap_timeout,
                                                                 code::imap_general_error,
                                                                 code::resolve_error };
            return not_connected_codes.count(ec);

        case errc::external_auth_invalid:
            static const std::set<error> auth_invalid_codes = { code::imap_response_no,
                                                                code::invalid_token };
            return auth_invalid_codes.count(ec);

        case errc::user_should_be_loaded:
            static const std::set<error> dont_load_codes = { code::no_auth_data, code::bad_karma };
            return !dont_load_codes.count(ec);

        default:
            return boost::system::error_category::equivalent(ec, condition);
        }
    }

    bool equivalent(int code, const boost::system::error_condition& condition) const
        noexcept override
    {
        return boost::system::error_category::equivalent(code, condition);
    }
};

const boost::system::error_category& error_category(code)
{
    static struct error_category instance;
    return instance;
}

const boost::system::error_category& get_error_condition_category()
{
    static error_condition_category instance;
    return instance;
}

}
