#pragma once

#include "types.h"

#include <string>
#include <vector>

namespace xeno {

enum class code : int
{
    ok = 0,
    environment_destructed_error,
    user_not_found,
    user_from_yandex,

    resolve_error,
    server_ip_blacklisted,
    imap_connect_error,
    imap_response_no,
    imap_response_bad,
    imap_timeout,
    imap_general_error,
    imap_protocol_disabled,
    imap_account_blocked,
    external_mailbox_exception,
    local_mailbox_exception,
    operation_exception,
    call_with_rc_exception,
    auth_tmp_error,

    store_message_error,
    so_error,
    sendbernar_error,
    auth_error,

    folder_not_found,
    folder_bad_status,
    message_not_found,
    label_not_found,
    threads_not_found,
    parent_not_found,
    symbol_not_found,

    need_clear_mailbox,
    need_check_timestamp,
    need_restart,
    need_parent_id,

    cannot_save_account,
    cannot_revoke_token,
    cannot_get_user_shard_id,
    cannot_get_attachments_sids,
    cannot_unsubscribe_user,
    cannot_sync_draft,

    no_auth_data,
    message_is_duplicate,
    smtp_connect_error,
    smtp_auth_error,
    smtp_check_email_error,
    smtp_send_perm_error,
    smtp_send_temp_error,
    body_is_empty,
    bad_request,
    message_is_spam,
    message_saved_with_error,
    message_too_big,

    // TODO remove later
    check_uidvalidity_error,

    invalid_token,
    get_access_token_error,
    get_refresh_token_error,
    token_cacher_error,
    bad_karma,
    mailish_id_too_big,

    COUNT
    // DO NOT FORGET TO EXTEND error_names
};

inline const std::string& error_message(code err)
{
    static constexpr std::initializer_list<const char*> names = {
        "success",
        "enviroment destructed without handler call",
        "user not found",
        "user from yandex",

        "resolve error",
        "server ip blacklisted",
        "imapclient connection error",
        "imapclient got NO in response",
        "imapclient got BAD in response",
        "imapclient timeout",
        "imapclient general error",
        "imapclient got protocol disabled",
        "imapclient got account blocked",
        "external mailbox exception",
        "local mailbox exception",
        "operation exception",
        "call with rc exception",
        "auth temporary error",

        "store message error",
        "so request error",
        "sendbernar request error",
        "auth error",

        "folder not found",
        "folder has bad status",
        "message not found",
        "label not found",
        "threads not found",
        "folder path is hierarchical but parent id is empty",
        "symbol not found",

        "need clear mailbox",
        "need check timestamp",
        "need restart",
        "need parent id",

        "can't save account",
        "can't revoke token",
        "can't get user shard id",
        "can't get attachments sids",
        "can't unsubscribe user",
        "can't sync draft",

        "no auth data",
        "can't store message: message is duplicate",
        "smtp connect error",
        "smtp auth error",
        "smtp check email error",
        "smtp send permanent error",
        "smtp send temporary error",
        "body is empty",
        "bad request",
        "message is spam",
        "message saved with error",
        "message too big",

        "check uidvalidity error",

        "invalid token",
        "get access token error",
        "get refresh token error",
        "token cacher error",
        "bad karma",
        "mailish_id too big"
    };
    static_assert(
        names.size() == static_cast<size_t>(code::COUNT),
        "Error codes count doesn't correspond with error names count");
    static const std::vector<std::string> error_names(names.begin(), names.end());

    return error_names[static_cast<int>(err)];
}

enum class errc
{
    ok = 0,
    imap_not_connected,
    perm_error,
    external_auth_invalid,
    user_should_be_loaded,

    COUNT
};

inline const std::string& errc_message(errc err)
{
    static constexpr std::initializer_list<const char*> names = { "success",
                                                                  "imap not connected",
                                                                  "error is permanent",
                                                                  "external auth invalid",
                                                                  "user should be loaded" };
    static_assert(
        names.size() == static_cast<size_t>(errc::COUNT),
        "Error conditions count doesn't correspond with condition names count");
    static const std::vector<std::string> condition_names(names.begin(), names.end());

    return condition_names[static_cast<int>(err)];
}

}

namespace yplatform {

template <>
struct is_error_enum<xeno::code>
{
    static const bool value = true;
};

}

namespace boost::system {

template <>
struct is_error_condition_enum<xeno::errc>
{
    static const bool value = true;
};

}

namespace xeno {

const boost::system::error_category& error_category(code);
const boost::system::error_category& get_error_condition_category();

inline boost::system::error_condition make_error_condition(errc e)
{
    return boost::system::error_condition(static_cast<int>(e), get_error_condition_category());
}

}
