#include <yplatform/util/split.h>

#include <boost/asio/ip/address.hpp>
#include <boost/asio/ip/network_v4.hpp>
#include <boost/asio/ip/network_v6.hpp>

namespace xeno {

using ip_address = boost::asio::ip::address;

inline auto make_ip_address(const std::string& str)
{
    return boost::asio::ip::make_address(str);
}

class ip_network
{
    using network_v4 = boost::asio::ip::network_v4;
    using network_v6 = boost::asio::ip::network_v6;

    const int V4_NETWORK_SIZE = 32;
    const int V6_NETWORK_SIZE = 128;

    enum class ip_version
    {
        v4,
        v6
    };

public:
    ip_network(const ip_address& addr, uint16_t size)
    {
        if (addr.is_v4())
        {
            version = ip_version::v4;
            v4 = network_v4(addr.to_v4(), size);
        }
        else
        {
            version = ip_version::v6;
            v6 = network_v6(addr.to_v6(), size);
        }
    }

    bool contains(const ip_address& addr)
    {
        return contains(ip_network(addr, addr.is_v4() ? V4_NETWORK_SIZE : V6_NETWORK_SIZE));
    }

    bool contains(const ip_network& network)
    {
        if (network.version != version) return false;
        return version == ip_version::v4 ? network.v4.is_subnet_of(v4) :
                                           network.v6.is_subnet_of(v6);
    }

private:
    ip_version version;
    network_v4 v4;
    network_v6 v6;
};

inline auto make_ip_network(const std::string& str)
{
    auto splitted = yplatform::util::split(str, "/");
    if (splitted.size() != 2)
    {
        throw std::runtime_error("bad ip network string: " + str);
    }

    auto addr = make_ip_address(splitted[0]);
    auto size = std::stoi(splitted[1]);
    return ip_network(addr, size);
}

}
