#pragma once

#include "settings.h"
#include <common/context.h>
#include <common/json.h>
#include <ymod_httpclient/cluster_client.h>
#include <yplatform/future/future.hpp>
#include <contrib/libs/jwt-cpp/include/jwt-cpp/jwt.h>

namespace xeno::encryption {

class cloud_iam_client
{
    using future_string = yplatform::future::future<string>;

public:
    cloud_iam_client(yplatform::reactor& reactor, const settings& settings) : settings(settings.iam)
    {
        auto conf = settings.http;
        conf.nodes = { settings.iam.node };
        client = boost::make_shared<yhttp::cluster_client>(reactor, conf);
    }

    string load_auth_token()
    {
        auto ctx = boost::make_shared<context>();
        auto resp = client->async_run(ctx, create_iam_request(), boost::asio::use_future).get();
        if (resp.status / 100 != 2)
        {
            throw std::runtime_error(
                "iam request: bad http status, status=" + std::to_string(resp.status) +
                ", body=" + resp.body);
        }
        return json::from_string(resp.body)["iamToken"].asString();
    }

private:
    yhttp::request create_iam_request()
    {
        json::value json;
        json["jwt"] = create_jwt_token();
        auto headers = "Content-Type: application/json\r\n";
        return yhttp::request::POST(settings.http_path, headers, json::to_string(json));
    }

    string create_jwt_token()
    {
        auto now = jwt::default_clock().now();
        return jwt::create()
            .set_key_id(settings.key_id)
            .set_issuer(settings.accout_id)
            .set_audience(settings.node + settings.http_path)
            .set_issued_at(now)
            .set_expires_at(now + settings.ttl)
            .sign(jwt::algorithm::ps256(settings.public_key, settings.private_key));
    }

    boost::shared_ptr<yhttp::cluster_client> client;
    iam_settings settings;
};

}
