#pragma once

#include "settings.h"
#include "cloud_iam_client.h"
#include <ymod_httpclient/cluster_client.h>

namespace xeno::encryption {

class cloud_kms_client
{
    boost::shared_ptr<yhttp::cluster_client> setup_client(
        yplatform::reactor& reactor,
        const settings& settings)
    {
        auto conf = settings.http;
        conf.nodes = { settings.kms.node };
        return boost::make_shared<yhttp::cluster_client>(reactor, conf);
    }

    string setup_auth_token(yplatform::reactor& reactor, const settings& settings)
    {
        cloud_iam_client iam(reactor, settings);
        return iam.load_auth_token();
    }

public:
    cloud_kms_client(yplatform::reactor& reactor, const settings& settings)
        : client(setup_client(reactor, settings))
        , auth_token(setup_auth_token(reactor, settings))
        , settings(settings.kms)
    {
    }

    string decrypt(const std::string& data)
    {
        auto ctx = boost::make_shared<context>();
        auto resp =
            client->async_run(ctx, create_decrypt_request(data), boost::asio::use_future).get();
        if (resp.status / 100 != 2)
        {
            throw std::runtime_error(
                "decrypt request: bad http status, status=" + std::to_string(resp.status) +
                ", body=" + resp.body);
        }
        return json::from_string(resp.body)["plaintext"].asString();
    }

private:
    yhttp::request create_decrypt_request(const string& data)
    {
        string path = "/kms/v1/keys/" + settings.key_id + ":decrypt";
        json::value json;
        json["ciphertext"] = data;
        auto headers =
            "Content-Type: application/json\r\nAuthorization: Bearer " + auth_token + "\r\n";
        return yhttp::request::POST(path, headers, json::to_string(json));
    }

    yplatform::reactor_ptr reactor;
    boost::shared_ptr<yhttp::cluster_client> client;
    string auth_token;
    kms_settings settings;
};

}
