#include "module.h"
#include "settings.h"
#include "cloud_kms_client.h"

namespace xeno::encryption {

yplatform::reactor_ptr setup_reactor()
{
    auto deleter = [](yplatform::reactor* r) {
        r->stop();
        r->fini();
        std::default_delete<yplatform::reactor>()(r);
    };
    auto res = boost::shared_ptr<yplatform::reactor>(new yplatform::reactor(), deleter);
    res->init(1, 1);
    res->run();
    return res;
}

versioned_keys prepare_keys(const settings& settings)
{
    if (settings.kms.enabled)
    {
        auto reactor = setup_reactor();
        cloud_kms_client client(*reactor, settings);
        versioned_keys res;
        for (auto&& [version, key] : settings.raw_keys)
        {
            auto decrypted_key = yplatform::base64_decode_str(client.decrypt(key));
            res.insert({ version, decrypted_key });
        }
        return res;
    }
    else
    {
        versioned_keys res;
        for (auto&& [version, key] : settings.raw_keys)
        {
            res.insert({ version, yplatform::base64_decode_str(key) });
        }
        return res;
    }
}

module::module(const yplatform::ptree& conf)
{
    settings settings;
    settings.parse_ptree(conf);
    keys = prepare_keys(settings);
}

string module::encrypt(const string& data, const string& iv)
{
    return blowfish::password::encrypt(data, keys, iv);
}

string module::decrypt(const string& data, const string& iv)
{
    return blowfish::password::decrypt(data, keys, iv);
}

}

#include <yplatform/module_registration.h>
REGISTER_MODULE(xeno::encryption::module);
