#include "cache_mailbox.h"
#include "folder.h"

#include <yplatform/util/sstream.h>
#include <yplatform/log.h>

#include <algorithm>

namespace xeno::mailbox {

cache_mailbox::cache_mailbox(const account_t& account, const yplatform::log::source& logger)
    : yplatform::log::contains_logger(logger), account_(account)
{
}

void cache_mailbox::update_folders_from_external(folder_vector_ptr folders)
{
    for (auto& ext_folder : *folders)
    {
        delim_ = ext_folder.path.delim;

        auto folder_it = folders_.find(ext_folder.path);
        if (folder_it != folders_.end())
        {
            auto& [path, cache_folder] = *folder_it;
            cache_folder.path.delim = delim_;
            update_folder_status_from_external(cache_folder, ext_folder);
        }
        else
        {
            create_folder_from_external(ext_folder);
        }
    }

    // TODO Вот тут надо проинициализировать папки у нас с именем как у нас в базе и создать
    // в удаленном имапе. Пока решено не создавать папки там
    // if (initial_folders_.size()) {}

    std::vector<path_t> to_delete;
    for (auto& [path, cache_folder] : folders_)
    {
        auto folder_it = std::find_if(
            folders->begin(),
            folders->end(),
            [& cache_folder = cache_folder](const folder& ext_folder) {
                return cache_folder.path == ext_folder.path;
            });

        if (folder_it == folders->end())
        {
            if (cache_folder.status == folder::status_t::to_create_local ||
                cache_folder.fid.empty())
            {
                to_delete.push_back(path);
            }
            else if (cache_folder.type != folder::type_t::user)
            {
                // TODO we should mark it until we will create local system folders in external
                // mailbox
                cache_folder.status = folder::status_t::sync_error;
            }
            else if (cache_folder.status != folder::status_t::to_clear_and_delete)
            {
                cache_folder.status = folder::status_t::to_delete;
            }
        }
    }

    for (auto& path : to_delete)
    {
        folders_.erase(path);
    }
}

void cache_mailbox::update_folder_status_from_external(
    folder& cache_folder,
    const folder& ext_folder)
{
    if (ext_folder.status == folder::status_t::to_create_external)
    {
        cache_folder.status = ext_folder.status;
    }
    else if (
        cache_folder.status == folder::status_t::to_create_external ||
        cache_folder.status == folder::status_t::sync_error)
    {
        if (is_not_inited_system_folder(ext_folder))
        {
            cache_folder = ext_folder;
            mark_to_init(cache_folder);
        }
        else
        {
            cache_folder.status =
                cache_folder.fid.size() ? folder::status_t::ok : folder::status_t::to_create_local;
        }
    }
    if (cache_folder.type != ext_folder.type)
    {
        cache_folder.status = folder::status_t::to_delete;
    }
}

error cache_mailbox::update_folder_info_from_external(const folder& ext_folder)
{
    bool correct_uidvalidity = true;
    auto it = folders_.find(ext_folder.path);
    if (it != folders_.end())
    {
        auto& [path, cache_folder] = *it;
        if (cache_folder.uidvalidity && cache_folder.uidvalidity != ext_folder.uidvalidity)
        {
            correct_uidvalidity = false;
            cache_folder.status = folder::status_t::to_update_and_clear;
        }
        else if (ext_folder.count == 0 && cache_folder.count != 0)
        {
            cache_folder.status = folder::status_t::to_update_and_clear;
        }
        else
        {
            cache_folder.uidvalidity = ext_folder.uidvalidity;
            cache_folder.count = ext_folder.count;
        }
        cache_folder.uidnext = ext_folder.uidnext;
        cache_folder.path.delim = ext_folder.path.delim;
    }

    return correct_uidvalidity ? code::ok : code::check_uidvalidity_error;
}

void cache_mailbox::create_folder_from_external(folder& ext_folder)
{
    if (ext_folder.type != folder::type_t::user)
    {
        if (is_not_inited_system_folder(ext_folder))
        {
            mark_to_init(ext_folder);
        }
        else if (is_changed_folder(ext_folder))
        {
            mark_to_update_and_clear(ext_folder);
        }
        else
        {
            mark_to_create_local(ext_folder);
        }
    }
    else if (ext_folder.status == folder::status_t::to_create_external)
    {
        folders_[ext_folder.path] = ext_folder;
    }
    else
    {
        mark_to_create_local(ext_folder);
    }
}

bool cache_mailbox::is_not_inited_system_folder(const folder& ext_folder) const
{
    for (auto& it : initial_folders_)
    {
        if (it.type == ext_folder.type)
        {
            return true;
        }
    }
    return false;
}

bool cache_mailbox::is_not_inited_system_folder(const fid_t& fid) const
{
    for (auto& it : initial_folders_)
    {
        if (it.fid == fid)
        {
            return true;
        }
    }
    return false;
}

void cache_mailbox::mark_to_init(folder& ext_folder)
{
    for (auto it = initial_folders_.begin(); it != initial_folders_.end(); ++it)
    {
        if (it->type == ext_folder.type)
        {
            ext_folder.fid = it->fid;
            ext_folder.status = folder::status_t::to_init;
            folders_[ext_folder.path] = ext_folder;
            // TODO move to sync folder operation
            initial_folders_.erase(it);
            break;
        }
    }
}

bool cache_mailbox::is_changed_folder(const folder& ext_folder) const
{
    for (auto& [path, cache_folder] : folders_)
    {
        if (cache_folder.type == ext_folder.type)
        {
            return true;
        }
    }
    return false;
}

void cache_mailbox::mark_to_update_and_clear(folder& ext_folder)
{
    for (auto& [path, cache_folder] : folders_)
    {
        if (cache_folder.type == ext_folder.type)
        {
            cache_folder.status = folder::status_t::to_update_and_clear;
            if (cache_folder.path != ext_folder.path)
            {
                cache_folder.path = ext_folder.path;
                folders_[cache_folder.path] = cache_folder;
                folders_.erase(path);
            }
            break;
        }
    }
}

void cache_mailbox::mark_to_create_local(folder& ext_folder)
{
    ext_folder.status = folder::status_t::to_create_local;
    folders_[ext_folder.path] = ext_folder;
}

folder_vector_ptr cache_mailbox::get_subfolders(const fid_t& parent_fid)
{
    auto subfolders = std::make_shared<folder_vector>();
    auto parent_folder = get_folder_by_fid(parent_fid);
    if (!parent_folder)
    {
        return subfolders;
    }
    for (auto& [path, folder] : folders_)
    {
        if (folder.is_subfolder_of(*parent_folder))
        {
            subfolders->push_back(folder);
        }
    }
    std::sort(subfolders->begin(), subfolders->end(), [](const folder& lhs, const folder& rhs) {
        return lhs.path.to_string() < rhs.path.to_string();
    });
    return subfolders;
}

void cache_mailbox::update_message_if_exists(const path_t& path, imap_id_t id, mid_t mid)
{
    auto folder_it = folders_.find(path);
    if (folder_it == folders_.end())
    {
        return;
    }
    auto& folder = folder_it->second;

    auto it = sync_newest_state_->folders.find(path);
    if (it != sync_newest_state_->folders.end())
    {
        auto& [folder_path, folder_state] = *it;

        auto msg_it = folder_state.messages_top.find(id);
        if (msg_it != folder_state.messages_top.end())
        {
            auto& [imap_id, message] = *msg_it;
            message.mid = mid;
            message.status = message::status_t::ok;

            if (id > folder.top_id)
            {
                folder.top_id = id;
            }

            if (folder.downloaded_range.empty())
            {
                folder.downloaded_range = imap_range{ id, id };
                if (!folder.top_id)
                {
                    folder.top_id = id;
                }
            }
        }
    }
}

void cache_mailbox::delete_messages(const path_t& path, imap_id_vector_ptr ids)
{
    auto it = sync_newest_state_->folders.find(path);
    if (it != sync_newest_state_->folders.end())
    {
        auto& [folder_path, folder_state] = *it;
        for (auto& id : *ids)
        {
            folder_state.messages_top.erase(id);
        }
    }
}

void cache_mailbox::delete_messages(const path_t& path, mid_vector_ptr mids)
{
    imap_id_vector_ptr ids = std::make_shared<imap_id_vector>();
    std::transform(mids->begin(), mids->end(), ids->begin(), [path, this](const mid_t& mid) {
        auto id_opt = get_imap_id_by_mid(path, mid);
        if (id_opt)
        {
            return *id_opt;
        }
        return imap_id_t{ 0 };
    });
    delete_messages(path, ids);
}

void cache_mailbox::move_messages(
    const path_t& from,
    const path_t& to,
    imap_id_transform_map_ptr transform_ids)
{
    auto from_it = sync_newest_state_->folders.find(from);
    auto to_it = sync_newest_state_->folders.find(to);
    if (from_it == sync_newest_state_->folders.end() || to_it == sync_newest_state_->folders.end())
    {
        return;
    }

    auto& [from_path, from_state] = *from_it;
    auto& [to_path, to_state] = *to_it;

    auto to_folder = get_folder_by_path(to);
    if (!to_folder)
    {
        return;
    }
    for (auto& [from_id, to_id] : *transform_ids)
    {
        auto it = from_state.messages_top.find(from_id);
        if (it != from_state.messages_top.end())
        {
            auto msg = std::move(it->second);
            from_state.messages_top.erase(it);

            msg.fid = to_folder->fid;
            msg.id = to_id;
            to_state.messages_top.emplace(to_id, std::move(msg));
        }
    }
}

account_t& cache_mailbox::account()
{
    return account_;
}

void cache_mailbox::set_account(account_t account)
{
    account_ = std::move(account);
}

void cache_mailbox::update_account(const account_t& account)
{
    account_.imap_ep = account.imap_ep;
    account_.smtp_ep = account.smtp_ep;

    auto& auth_data = account_.auth_data;
    for (auto& data : account.auth_data)
    {
        auto it = std::find(auth_data.begin(), auth_data.end(), data);
        if (it != auth_data.end())
        {
            auth_data.erase(it);
        }
        auth_data.push_back(data);
    }
}

void cache_mailbox::invalidate_auth_data(const auth_data& data)
{
    auto& auth_data = account_.auth_data;
    auto it = std::find(auth_data.begin(), auth_data.end(), data);
    if (it != auth_data.end())
    {
        auth_data.erase(it);
    }
}

void cache_mailbox::set_karma(const karma_t& karma)
{
    account_.karma = karma;
}

const cache_mailbox::path_folder_map& cache_mailbox::folders() const
{
    return folders_;
}

folder_vector_ptr cache_mailbox::folders_copy() const
{
    auto result_vector = std::make_shared<folder_vector>();
    for (auto& [path, folder] : folders_)
    {
        result_vector->push_back(folder);
    }
    return result_vector;
}

void cache_mailbox::update_folders_from_local(folder_vector_ptr folders)
{
    for (const auto& local_folder : *folders)
    {
        if (local_folder.path.empty()) continue;
        auto folder_it = folders_.find(local_folder.path);
        if (folder_it != folders_.end())
        {
            auto& [path, cache_folder] = *folder_it;
            if (cache_folder.status == folder::status_t::to_create_local)
            {
                cache_folder.status = folder::status_t::ok;
            }
            cache_folder.fid = local_folder.fid;
            cache_folder.top_id = local_folder.top_id;
            cache_folder.downloaded_range = local_folder.downloaded_range;
        }
        else
        {
            folders_[local_folder.path] = local_folder;
        }
    }
}

void cache_mailbox::set_initial_folders(folder_vector_ptr folders)
{
    for (auto& folder : *folders)
    {
        if (folder.type != folder::type_t::user && folder.path.empty())
        {
            initial_folders_.push_back(folder);
        }
    }
}

void cache_mailbox::mark_inited(const path_t& path)
{
    auto folder_it = folders_.find(path);
    if (folder_it != folders_.end())
    {
        auto& [path, folder] = *folder_it;
        folder.status = folder::status_t::ok;
    }
}

void cache_mailbox::clear_folders()
{
    folders_.clear();
}

void cache_mailbox::clear_folder(const path_t& path)
{
    auto folder_it = folders_.find(path);
    if (folder_it == folders_.end())
    {
        return;
    }

    auto& [folder_path, folder] = *folder_it;
    auto it = sync_newest_state_->folders.find(path);
    if (it != sync_newest_state_->folders.end())
    {
        it->second.messages_top.clear();
    }
    folder.top_id = 0;
    folder.count = 0;
    folder.uidvalidity = 0;
    folder.downloaded_range = imap_range{ 0, 0 };
}

void cache_mailbox::delete_folder_by_fid(const fid_t& fid)
{
    auto path = get_path_by_fid(fid);
    if (!path.empty())
    {
        delete_folder_by_path(path);
    }
}

void cache_mailbox::delete_folder_by_path(const path_t& path)
{
    auto folder_it = folders_.find(path);
    if (folder_it != folders_.end())
    {
        auto& [path, folder_to_delete] = *folder_it;
        if (!folder_to_delete.childs.empty())
        {
            for (const auto& child : folder_to_delete.childs)
            {
                delete_folder_by_path(child);
            }
        }

        if (folder_to_delete.type != folder::type_t::user)
        {
            auto system_folder_it = std::find_if(
                initial_folders_.begin(),
                initial_folders_.end(),
                [type = folder_to_delete.type](const folder& system_folder) {
                    return type == system_folder.type;
                });
            if (system_folder_it != initial_folders_.end())
            {
                if (system_folder_it->fid == folder_to_delete.fid)
                {
                    YLOG_L(error)
                        << "inconsistent data in cache: initial_folders already contains " +
                            folder_to_delete.to_string(folder_to_delete.type);
                }
                else
                {
                    throw std::runtime_error(
                        "2 different folders with type: " +
                        folder_to_delete.to_string(folder_to_delete.type));
                }
            }
            else
            {
                initial_folders_.emplace_back();
                initial_folders_.back().type = folder_to_delete.type;
                initial_folders_.back().fid = folder_to_delete.fid;
            }
        }
        sync_newest_state_->folders.erase(path);
        folders_.erase(path);
    }
}

void cache_mailbox::rename_folder(
    const path_t& old_path,
    const path_t& new_path,
    const fid_t_opt& new_parent)
{
    auto it = folders_.find(old_path);
    if (it == folders_.end())
    {
        return;
    }

    auto [path, folder] = *it;
    folder.path = new_path;

    childs_path_vector new_children;
    for (auto& child : folder.childs)
    {
        auto new_child_path = new_path.make_child_path(child.get_name());
        rename_folder(child, new_child_path, fid_t_opt());
        new_children.push_back(new_child_path);
    }
    folder.childs = new_children;

    folders_.erase(it);
    folders_.emplace(new_path, folder);

    if (new_parent)
    {
        auto old_parent_path = old_path.get_parent_path();
        auto old_parent_it = folders_.find(old_parent_path);
        if (old_parent_it != folders_.end())
        {
            auto& children = old_parent_it->second.childs;
            auto new_end = std::remove(children.begin(), children.end(), old_path);
            children.erase(new_end, children.end());
        }

        auto new_parent_path = new_path.get_parent_path();
        auto new_parent_it = folders_.find(new_parent_path);
        if (new_parent_it != folders_.end())
        {
            new_parent_it->second.childs.push_back(new_path);
        }
    }
}

path_t cache_mailbox::get_path_by_fid(const fid_t& fid)
{
    if (fid.size())
    {
        auto folder_it = std::find_if(
            folders_.begin(), folders_.end(), [&fid](const std::pair<const path_t, folder>& pair) {
                auto& [path, cache_folder] = pair;
                return cache_folder.fid == fid;
            });
        if (folder_it != folders_.end())
        {
            auto& [path, cache_folder] = *folder_it;
            return cache_folder.path;
        }
    }

    path_t res;
    res.delim = delim_;
    return res;
}

path_t_opt cache_mailbox::get_path_by_folder_type(folder::type_t type)
{
    for (auto& [path, folder] : folders_)
    {
        if (folder.type == type)
        {
            return folder.path;
        }
    }
    return path_t_opt{};
}

folder_opt cache_mailbox::get_folder_by_type(folder::type_t type)
{
    for (auto& [path, folder] : folders_)
    {
        if (folder.type == type)
        {
            return folder;
        }
    }
    return folder_opt{};
}

void cache_mailbox::update_downloaded_range(const path_t& path, const imap_range& downloaded)
{
    auto folder_it = folders_.find(path);
    if (folder_it != folders_.end())
    {
        auto& [path, folder] = *folder_it;
        if (folder.downloaded_range.empty())
        {
            folder.downloaded_range = downloaded;
        }
        else if (downloaded.bottom() == folder.downloaded_range.top())
        {
            folder.downloaded_range =
                imap_range(downloaded.top(), folder.downloaded_range.bottom());
        }
        else if (downloaded.top() == folder.downloaded_range.bottom())
        {
            folder.downloaded_range =
                imap_range(folder.downloaded_range.top(), downloaded.bottom());
        }
    }
}

imap_range_opt cache_mailbox::get_downloaded_range(const path_t& path)
{
    auto folder_it = folders_.find(path);
    if (folder_it != folders_.end())
    {
        auto& [path, folder] = *folder_it;
        if (!folder.downloaded_range.empty())
        {
            return folder.downloaded_range;
        }
    }
    return imap_range_opt{};
}

folder_opt cache_mailbox::get_folder_by_fid(const fid_t& fid)
{
    for (auto& [path, folder] : folders_)
    {
        if (folder.fid == fid)
        {
            return folder;
        }
    }
    return folder_opt();
}

folder_opt cache_mailbox::get_folder_by_path(const path_t& path)
{
    auto it = folders_.find(path);
    return it != folders_.end() ? it->second : folder_opt();
}

fid_t_opt cache_mailbox::get_parent_fid_by_child_path(const path_t& child_path)
{
    auto parent_path = child_path.get_parent_path();
    if (!parent_path.empty())
    {
        return get_fid_by_path(parent_path);
    }
    return {};
}

void cache_mailbox::set_folder_status_by_path(const path_t& path, folder::status_t status)
{
    auto it = folders_.find(path);
    if (it != folders_.end())
    {
        auto& [path, folder] = *it;
        folder.status = status;
    }
}

void cache_mailbox::set_folder_status_by_fid(const fid_t& fid, folder::status_t status)
{
    for (auto& [path, folder] : folders_)
    {
        if (folder.fid == fid)
        {
            folder.status = status;
            break;
        }
    }
}

folder& cache_mailbox::get_folder(const path_t& path)
{
    auto it = folders_.find(path);
    if (it == folders_.end())
    {
        throw std::runtime_error("folder not found " + path.to_string());
    }

    return it->second;
}

folder_opt cache_mailbox::get_folder_by_status(folder::status_t status) const
{
    auto matcher = [status](const std::pair<path_t, folder>& pair) {
        auto& [path, folder] = pair;
        return folder.status == status;
    };

    if (status == folder::status_t::to_delete)
    {
        // reverse order required for deletion
        auto it = std::find_if(folders_.rbegin(), folders_.rend(), matcher);
        if (it != folders_.rend())
        {
            return it->second;
        }
    }
    else
    {
        auto it = std::find_if(folders_.begin(), folders_.end(), matcher);
        if (it != folders_.end())
        {
            return it->second;
        }
    }
    return folder_opt();
}

path_fid_vector_ptr cache_mailbox::get_paths_to_check_timestamps() const
{
    auto to_ok = std::make_shared<path_fid_vector>();

    for (auto& [path, folder] : folders_)
    {
        if (folder.status == mailbox::folder::status_t::ok && folder.count)
        {
            to_ok->emplace_back(path_fid_pair(folder.path, folder.fid));
        }
    }

    return to_ok;
}

fid_t_opt cache_mailbox::get_fid_by_path(const path_t& path) const
{
    auto folder_it = folders_.find(path);
    if (folder_it != folders_.end())
    {
        auto& [folder_path, folder] = *folder_it;
        return folder.fid;
    }
    return fid_t_opt{};
}

imap_id_opt cache_mailbox::get_imap_id_by_mid(const path_t& path, mid_t mid) const
{
    auto it = sync_newest_state_->folders.find(path);
    if (it != sync_newest_state_->folders.end())
    {
        auto& [folder_path, folder_state] = *it;
        for (auto& [imap_id, msg] : folder_state.messages_top)
        {
            if (msg.mid == mid)
            {
                return msg.id;
            }
        }
    }
    return imap_id_opt{};
}

std::time_t cache_mailbox::get_last_sync_ts() const
{
    return account_.last_sync_ts;
}

void cache_mailbox::update_message_errors_count(const path_t& path, imap_id_t id, uint32_t count)
{
    auto it = sync_newest_state_->folders.find(path);
    if (it != sync_newest_state_->folders.end())
    {
        auto& [folder_path, folder_state] = *it;

        auto msg_it = folder_state.messages_top.find(id);
        if (msg_it != folder_state.messages_top.end())
        {
            auto& [imap_id, message] = *msg_it;
            message.errors_count = count;
        }
    }
}

void cache_mailbox::save_message_errors_count(const path_t& path, imap_id_t id)
{
    auto it = sync_newest_state_->folders.find(path);
    if (it == sync_newest_state_->folders.end())
    {
        return;
    }

    auto& [folder_path, folder_state] = *it;
    auto msg_it = folder_state.messages_top.find(id);
    if (msg_it == folder_state.messages_top.end())
    {
        return;
    }

    auto& [imap_id, message] = *msg_it;
    message.saved_errors_count = message.errors_count;
}

imap_id_opt cache_mailbox::get_min_imap_id_from_top(const path_t& path)
{
    auto folder = get_folder_by_path(path);
    if (!folder)
    {
        return imap_id_opt{};
    }
    auto min_imap_id_from_top = folder->top_id ? folder->top_id : folder->uidnext - 1;

    auto it = sync_newest_state_->folders.find(path);
    if (it != sync_newest_state_->folders.end())
    {
        auto& [folder_path, folder_state] = *it;
        for (auto& [imap_id, message] : folder_state.messages_top)
        {
            if (message.id < min_imap_id_from_top && message.status == message::status_t::ok)
            {
                min_imap_id_from_top = message.id;
            }
        }
    }
    return min_imap_id_from_top;
}

bool cache_mailbox::has_security_lock()
{
    return account_.has_security_lock();
}

void cache_mailbox::erase_security_lock()
{
    for (auto& data : account_.auth_data)
    {
        data.security_lock = false;
    }
}

message_vector_ptr cache_mailbox::get_oldest_messages_for_sync(const path_t& path)
{
    auto it = oldest_messages_for_sync_.find(path);
    if (it != oldest_messages_for_sync_.end())
    {
        auto [path, messages] = *it;
        return messages;
    }
    return {};
}

redownload_messages_state_ptr cache_mailbox::redownload_messages_state()
{
    return redownload_messages_state_;
}

sync_newest_state_ptr cache_mailbox::sync_newest_state()
{
    return sync_newest_state_;
}

sync_oldest_flags_and_deleted_state_ptr cache_mailbox::sync_oldest_flags_and_deleted_state()
{
    return sync_oldest_flags_and_deleted_state_;
}

void cache_mailbox::update_oldest_messages_for_sync(const path_t& path, message_vector_ptr messages)
{
    oldest_messages_for_sync_[path] = messages;
    oldest_processed_range_.erase(path);
}

imap_range cache_mailbox::get_oldest_processed_range(const path_t& path)
{
    if (!oldest_processed_range_.count(path))
    {
        oldest_processed_range_[path] = get_folder(path).downloaded_range;
    }
    return oldest_processed_range_[path];
}

void cache_mailbox::update_oldest_processed_range(const path_t& path, const imap_range& range)
{
    oldest_processed_range_[path] = range;
}

yplatform::ptree cache_mailbox::dump()
{
    yplatform::ptree dump;
    dump.put_child("account", account_.dump());

    if (folders_.size())
    {
        yplatform::ptree folders;
        for (auto& [path, folder] : folders_)
        {
            auto folder_dump = folder.dump();

            auto it = sync_newest_state_->folders.find(path);
            if (it != sync_newest_state_->folders.end())
            {
                yplatform::ptree top_tree;
                auto& [folder_path, folder_state] = *it;
                for (auto& [imap_id, msg] : folder_state.messages_top)
                {
                    top_tree.push_back(std::make_pair("", msg.dump()));
                }
                folder_dump.put_child("messages_top", top_tree);
            }
            folders.push_back(std::make_pair("folder", folder_dump));
        }
        dump.put_child("folders", folders);
    }

    return dump;
}

}
