#pragma once

#include "folder.h"
#include "flags.h"
#include "message.h"
#include "redownload_messages_state.h"
#include "sync_newest_state.h"
#include "sync_oldest_flags_and_deleted_state.h"

#include <common/account.h>
#include <common/karma.h>
#include <common/errors.h>

#include <yplatform/ptree.h>

#include <memory>
#include <unordered_map>

namespace xeno::mailbox {

class cache_mailbox : public yplatform::log::contains_logger
{
public:
    using path_folder_map = std::map<path_t, folder>;
    using path_messages_map = std::map<path_t, message_vector_ptr>;
    using path_range_map = std::map<path_t, imap_range>;

    cache_mailbox() = default;
    cache_mailbox(const account_t& account, const yplatform::log::source& logger);

    void update_folders_from_external(folder_vector_ptr folders);
    error update_folder_info_from_external(const folder& ext_folder);
    void update_folders_from_local(folder_vector_ptr folders);
    void delete_folder_by_fid(const fid_t& fid);
    void delete_folder_by_path(const path_t& path);

    void rename_folder(const path_t& old_path, const path_t& new_path, const fid_t_opt& new_parent);

    path_t get_path_by_fid(const fid_t& fid);
    path_t_opt get_path_by_folder_type(folder::type_t type);
    folder_opt get_folder_by_type(folder::type_t type);
    folder_opt get_folder_by_fid(const fid_t& fid);
    folder_opt get_folder_by_path(const path_t& path);
    fid_t_opt get_parent_fid_by_child_path(const path_t& child_path);
    folder& get_folder(const path_t& path);
    folder_vector_ptr get_subfolders(const fid_t& parent_fid);

    void set_folder_status_by_path(const path_t& path, folder::status_t status);
    void set_folder_status_by_fid(const fid_t& fid, folder::status_t status);

    void clear_folders();
    void clear_folder(const path_t& path);

    void update_message_if_exists(const path_t& path, imap_id_t id, mid_t mid);

    void delete_messages(const path_t& path, imap_id_vector_ptr ids);
    void delete_messages(const path_t& path, mid_vector_ptr mids);
    void move_messages(
        const path_t& from,
        const path_t& to,
        imap_id_transform_map_ptr transform_ids);

    void update_downloaded_range(const path_t& path, const imap_range& downloaded);
    imap_range_opt get_downloaded_range(const path_t& path);

    message_vector_ptr get_oldest_messages_for_sync(const path_t& path);
    void update_oldest_messages_for_sync(const path_t& path, message_vector_ptr messages);
    imap_range get_oldest_processed_range(const path_t& path);
    void update_oldest_processed_range(const path_t& path, const imap_range& range);
    imap_id_opt get_min_imap_id_from_top(const path_t& path);

    fid_t_opt get_fid_by_path(const path_t& path) const;
    imap_id_opt get_imap_id_by_mid(const path_t& path, mid_t mid) const;

    account_t& account();
    void set_account(account_t account);
    void update_account(const account_t& account);

    void invalidate_auth_data(const auth_data& data);
    void set_karma(const karma_t& karma);

    const path_folder_map& folders() const;

    folder_vector_ptr folders_copy() const;
    void set_initial_folders(folder_vector_ptr folders);
    void mark_inited(const path_t& path);

    folder_opt get_folder_by_status(folder::status_t status) const;

    path_fid_vector_ptr get_paths_to_check_timestamps() const;

    redownload_messages_state_ptr redownload_messages_state();
    sync_newest_state_ptr sync_newest_state();
    sync_oldest_flags_and_deleted_state_ptr sync_oldest_flags_and_deleted_state();

    void update_message_errors_count(const path_t& path, imap_id_t id, uint32_t count);
    void save_message_errors_count(const path_t& path, imap_id_t id);

    std::time_t get_last_sync_ts() const;

    bool has_security_lock();
    void erase_security_lock();

    yplatform::ptree dump();

    bool is_not_inited_system_folder(const fid_t& fid) const;

protected:
    void update_folder_status_from_external(folder& cache_folder, const folder& ext_folder);
    void create_folder_from_external(folder& ext_folder);

    bool is_not_inited_system_folder(const folder& ext_folder) const;
    bool is_changed_folder(const folder& ext_folder) const;

    void mark_to_init(folder& ext_folder);
    void mark_to_update_and_clear(folder& ext_folder);
    void mark_to_create_local(folder& ext_folder);

private:
    account_t account_;
    path_folder_map folders_;
    path_messages_map oldest_messages_for_sync_;
    path_range_map oldest_processed_range_;
    folder_vector initial_folders_;
    redownload_messages_state_ptr redownload_messages_state_ =
        std::make_shared<mailbox::redownload_messages_state>();
    sync_newest_state_ptr sync_newest_state_ = std::make_shared<mailbox::sync_newest_state>();
    sync_oldest_flags_and_deleted_state_ptr sync_oldest_flags_and_deleted_state_ =
        std::make_shared<mailbox::sync_oldest_flags_and_deleted_state>();

    char delim_;
};

using cache_mailbox_ptr = std::shared_ptr<cache_mailbox>;
}
