#include "flags.h"

#include <yplatform/util/sstream.h>

#include <boost/algorithm/string/join.hpp>
#include <vector>

namespace xeno::mailbox {
flags_t::flags_t(const system_flags_t& system, const user_flags_t& user)
    : system_flags(system), user_flags(user)
{
}

yplatform::ptree flags_t::dump()
{
    yplatform::ptree dump;

    if (system_flags.size())
    {
        std::vector<std::string> flags;
        for (auto& flag : system_flags)
        {
            flags.push_back(to_string(flag));
        }
        dump.put("system_flags", boost::algorithm::join(flags, ", "));
    }

    if (user_flags.size())
    {
        dump.put("user_flags", boost::algorithm::join(user_flags, ", "));
    }

    return dump;
}

std::string flags_t::to_string(system_flag_t flag)
{
    switch (flag)
    {
    case system_flag_t::seen:
        return "seen";
    case system_flag_t::answered:
        return "answered";
    case system_flag_t::flagged:
        return "flagged";
    case system_flag_t::deleted:
        return "deleted";
    case system_flag_t::draft:
        return "draft";
    case system_flag_t::recent:
        return "recent";
    default:
        return "undefined status";
    }
}

static std::map<std::string, system_flag_t> flags = {
    { "seen", system_flag_t::seen },       { "answered", system_flag_t::answered },
    { "flagged", system_flag_t::flagged }, { "deleted", system_flag_t::deleted },
    { "draft", system_flag_t::draft },     { "recent", system_flag_t::recent }
};

system_flag_t flags_t::from_string(const std::string& flag)
{
    auto it = flags.find(flag);
    if (it != flags.end())
    {
        return it->second;
    }
    throw std::runtime_error("Flag conversion error: no flag for string \"" + flag + "\"");
}

bool flags_t::operator==(const flags_t& other) const
{
    return std::tie(system_flags, user_flags) == std::tie(other.system_flags, other.user_flags);
}

bool flags_t::operator!=(const flags_t& other) const
{
    return !(*this == other);
}

namespace mock_helpers {

void mark_flags(flags_t& change, const flags_t& add, const flags_t& del)
{
    change.system_flags.insert(add.system_flags.begin(), add.system_flags.end());
    change.user_flags.insert(add.user_flags.begin(), add.user_flags.end());

    for (auto& flag : del.system_flags)
    {
        change.system_flags.erase(flag);
    }

    for (auto& flag : del.user_flags)
    {
        change.user_flags.erase(flag);
    }
}

}
}
