#pragma once

#include <yplatform/ptree.h>

#include <map>
#include <memory>
#include <string>
#include <vector>
#include <utility>
#include <set>

namespace xeno::mailbox {

using lid = std::string;
using lid_vector = std::vector<lid>;

enum class system_flag_t : int32_t
{
    seen = 0,
    answered,
    flagged,
    deleted,
    draft,
    recent,

    // This is only yandex flag, not from RFC
    // forwarded,

    COUNT
};

inline const std::string& system_flag_name(system_flag_t flag)
{
    static constexpr std::initializer_list<const char*> names = { "\\Seen",    "\\Answered",
                                                                  "\\Flagged", "\\Deleted",
                                                                  "\\Draft",   "\\Recent" };
    static_assert(
        names.size() == int(system_flag_t::COUNT),
        "System flags count doesn't correspond with flag names count");
    static const std::vector<std::string> flag_names(names.begin(), names.end());

    return flag_names[int(flag)];
}

class flags_t
{
public:
    using system_flags_t = std::set<system_flag_t>;
    using user_flags_t = std::set<std::string>;

    flags_t() = default;
    flags_t(const system_flags_t& system, const user_flags_t& user);

    system_flags_t system_flags;
    user_flags_t user_flags;

    yplatform::ptree dump();

    static std::string to_string(system_flag_t flag);
    static system_flag_t from_string(const std::string& flag);
    bool operator==(const flags_t& other) const;
    bool operator!=(const flags_t& other) const;
};

using flags_ptr = std::shared_ptr<flags_t>;

namespace mock_helpers {

template <typename Flags, typename Flag>
void add_flag(Flags& flags, Flag flag);

template <typename Flags, typename Flag>
void del_flag(Flags& flags, Flag flag);

void mark_flags(flags_t& change, const flags_t& add, const flags_t& del);

}
}
