#include "folder.h"

#include <yplatform/util/split.h>

#include <boost/algorithm/string/join.hpp>

namespace xeno::mailbox {

path_t path_t::get_parent_path() const
{
    std::vector<std::string> folders_hierarchy;
    yplatform::util::split(folders_hierarchy, path, std::string{ delim });
    path_t parent;
    parent.delim = delim;
    if (folders_hierarchy.size() > 1)
    {
        folders_hierarchy.erase(std::next(folders_hierarchy.rbegin()).base());
        parent.path = boost::algorithm::join(folders_hierarchy, std::string{ delim });
    }
    return parent;
}

path_t path_t::make_child_path(const std::string& child_name) const
{
    if (empty())
    {
        return path_t(child_name, delim);
    }

    return path_t(path + std::string{ delim } + child_name, delim);
}

std::string path_t::get_name() const
{
    std::vector<std::string> folders_hierarchy;
    yplatform::util::split(folders_hierarchy, path, std::string{ delim });
    if (folders_hierarchy.empty())
    {
        return "";
    }
    return folders_hierarchy.back();
}

imap_range::imap_range(imap_id_t top, imap_id_t bottom)
    : top_(std::max(top, bottom)), bottom_(std::min(top, bottom))
{
}

imap_range::imap_range() : top_(0), bottom_(0)
{
}

bool imap_range::within(const imap_id_t& id) const
{
    return bottom_ <= id && id <= top_;
}

imap_id_t imap_range::top() const
{
    return top_;
}

imap_id_t imap_range::bottom() const
{
    return bottom_;
}

folder::folder(
    const path_t& path,
    const fid_t& fid,
    uint64_t uidvalidity,
    imap_id_t uidnext,
    num_t count)
    : path(path), fid(fid), count(count), uidvalidity(uidvalidity), uidnext(uidnext)
{
}

bool folder::is_subfolder_of(const folder& parent) const
{
    return path.to_string().find(parent.path.to_string() + std::string{ parent.path.delim }) == 0;
}

folder::type_t folder::type_from_string(const std::string& type)
{
    if (type == "inbox")
    {
        return type_t::inbox;
    }
    else if (type == "sent")
    {
        return type_t::sent;
    }
    else if (type == "trash")
    {
        return type_t::trash;
    }
    else if (type == "spam")
    {
        return type_t::spam;
    }
    else if (type == "drafts")
    {
        return type_t::drafts;
    }
    else if (type == "outbox")
    {
        return type_t::outbox;
    }
    else if (type == "user")
    {
        return type_t::user;
    }
    else
    {
        throw std::runtime_error("undefined type:" + type);
    }
}

yplatform::ptree folder::dump()
{
    yplatform::ptree dump;
    dump.put("path", path.to_string());
    dump.put("delim", std::to_string(path.delim));
    dump.put("fid", fid);
    dump.put("status", to_string(status));
    dump.put("sync_status", to_string(sync_status));
    dump.put("type", to_string(type));
    dump.put("importance", to_string(importance));
    dump.put("api_read_lock", api_read_lock ? "true" : "false");
    dump.put("count", count);
    dump.put("uidvalidity", uidvalidity);
    dump.put(
        "downloaded_range",
        (std::to_string(downloaded_range.top()) + ":" + std::to_string(downloaded_range.bottom())));

    if (childs.size())
    {
        for (auto& child : childs)
        {
            yplatform::ptree child_tree;
            child_tree.put("child", child.to_string());
            dump.push_back(std::make_pair("", child_tree));
        }
    }

    return dump;
}

const std::string& folder::to_string(status_t status)
{
    static constexpr std::initializer_list<const char*> names = {
        "ok",      "sync error",          "to delete", "to create local",    "to create external",
        "to init", "to update and clear", "to clear",  "to clear and delete"
    };
    static_assert(
        names.size() == static_cast<size_t>(status_t::COUNT),
        "Folder status count doesn't correspond with status names count");
    static const std::vector<std::string> status_names(names.begin(), names.end());

    return status_names[static_cast<int>(status)];
}

const std::string& folder::to_string(importance_t importance)
{
    static constexpr std::initializer_list<const char*> names = { "inbox",
                                                                  "important",
                                                                  "not_important" };
    static_assert(
        names.size() == static_cast<size_t>(importance_t::COUNT),
        "Folder importance count doesn't correspond with importance names count");
    static const std::vector<std::string> importance_names(names.begin(), names.end());

    return importance_names[static_cast<int>(importance)];
}

const std::string& folder::to_string(type_t type)
{
    static constexpr std::initializer_list<const char*> names = {
        "inbox", "user", "sent", "drafts", "trash", "spam", "outbox",
    };
    static_assert(
        names.size() == static_cast<size_t>(type_t::COUNT),
        "Folder type count doesn't correspond with type names count");
    static const std::vector<std::string> type_names(names.begin(), names.end());

    return type_names[static_cast<int>(type)];
}

const std::string& folder::to_string(sync_status_t sync_status)
{
    static constexpr std::initializer_list<const char*> names = { "reqular", "error" };
    static_assert(
        names.size() == static_cast<size_t>(sync_status_t::COUNT),
        "Folder sync status count doesn't correspond with type names count");
    static const std::vector<std::string> type_names(names.begin(), names.end());

    return type_names[static_cast<int>(sync_status)];
}

}
