#pragma once

#include "message.h"

#include <yplatform/ptree.h>
#include <string>
#include <vector>
#include <unordered_set>
#include <utility>
#include <boost/optional.hpp>

namespace xeno::mailbox {

using fid_t_ptr = std::shared_ptr<fid_t>;
using fid_t_opt = boost::optional<fid_t>;
using fid_vector = std::vector<fid_t>;
using fid_vector_ptr = std::shared_ptr<fid_vector>;

struct path_t
{
    path_t() = default;
    path_t(const std::string& path, char delim) : path(path), delim(delim)
    {
    }

    path_t get_parent_path() const;
    path_t make_child_path(const std::string& child_name) const;
    std::string get_name() const;

    bool operator==(const path_t& other) const
    {
        return path == other.path;
    }

    bool operator!=(const path_t& other) const
    {
        return !(*this == other);
    }

    bool operator<(const path_t& other) const
    {
        return path < other.path;
    }

    bool operator<=(const path_t& other) const
    {
        return path <= other.path;
    }

    bool operator>=(const path_t& other) const
    {
        return path >= other.path;
    }

    bool operator>(const path_t& other) const
    {
        return path > other.path;
    }

    bool empty() const
    {
        return path.empty();
    }

    const std::string& to_string() const
    {
        return path;
    }

    std::string path;
    char delim = 0;
};
using path_t_opt = boost::optional<path_t>;
using path_t_ptr = std::shared_ptr<path_t>;
using path_vector = std::vector<path_t>;
using path_vector_ptr = std::shared_ptr<path_vector>;

using path_fid_pair = std::pair<path_t, fid_t>;
using path_fid_vector = std::vector<path_fid_pair>;
using path_fid_vector_ptr = std::shared_ptr<path_fid_vector>;

struct folder;
using folder_opt = boost::optional<folder>;
using folder_ptr = std::shared_ptr<folder>;
using folder_vector = std::vector<folder>;
using folder_vector_ptr = std::shared_ptr<folder_vector>;
using childs_path_vector = std::vector<path_t>;

class imap_range
{
public:
    imap_range(imap_id_t top, imap_id_t bottom);
    imap_range();

    bool operator==(const imap_range& other)
    {
        return std::tie(top_, bottom_) == std::tie(other.top_, other.bottom_);
    }

    bool operator!=(const imap_range& other)
    {
        return std::tie(top_, bottom_) != std::tie(other.top_, other.bottom_);
    }

    bool within(const imap_id_t& id) const;
    bool empty() const
    {
        return !top_ || !bottom_;
    }

    imap_id_t top() const;
    imap_id_t bottom() const;

private:
    imap_id_t top_;
    imap_id_t bottom_;
};
using imap_range_opt = boost::optional<imap_range>;

struct folder
{
public:
    folder() = default;
    folder(
        const path_t& path,
        const fid_t& fid = "",
        uint64_t uidvalidity = 0,
        imap_id_t uidnext = 1,
        num_t count = 0);

    enum class status_t : int
    {
        ok = 0,
        sync_error,
        to_delete,
        to_create_local,
        to_create_external,
        to_init,
        to_update_and_clear,
        to_clear,
        to_clear_and_delete,

        COUNT
        // DO NOT FORGET TO EXTEND status_names
    };
    enum class sync_status_t : int
    {
        reqular = 0,
        error,

        COUNT
        // DO NOT FORGET TO EXTEND status_names
    };
    enum class type_t : int
    {
        inbox = 0,
        user,
        sent,
        drafts,
        trash,
        spam,
        outbox,

        COUNT
        // DO NOT FORGET TO EXTEND type_names
    };
    enum class importance_t : int
    {
        inbox = 0,
        important,
        not_important,

        COUNT
        // DO NOT FORGET TO EXTEND importance_names
    };

    static folder::type_t type_from_string(const std::string& type);
    bool is_subfolder_of(const folder& parent) const;

    path_t path;
    fid_t fid;
    childs_path_vector childs;
    status_t status{ status_t::ok };
    sync_status_t sync_status{ sync_status_t::reqular };
    type_t type{ type_t::user };
    importance_t importance{ importance_t::important };
    bool api_read_lock{ true };
    num_t count{ 0 };
    uint64_t uidvalidity{ 0 };
    imap_id_t uidnext{ 0 };
    imap_id_t top_id{ 0 };
    imap_range downloaded_range{ 0, 0 };

    yplatform::ptree dump();
    static const std::string& to_string(status_t status);
    static const std::string& to_string(sync_status_t sync_status);
    static const std::string& to_string(importance_t importance);
    static const std::string& to_string(type_t type);
};

}

namespace std {

template <>
struct hash<xeno::mailbox::path_t>
{
    std::size_t operator()(const xeno::mailbox::path_t& k) const
    {
        using std::hash;
        using std::string;

        return (hash<string>()(k.to_string()));
    };
};

}
