#pragma once

#include "flags.h"

#include <yplatform/time_traits.h>
#include <yplatform/ptree.h>

#include <list>
#include <vector>
#include <utility>
#include <unordered_set>
#include <unordered_map>
#include <boost/optional.hpp>

namespace xeno::mailbox {

using imap_id_t = uint32_t;
using imap_id_opt = boost::optional<imap_id_t>;
using imap_id_vector = std::vector<imap_id_t>;
using imap_id_vector_ptr = std::shared_ptr<imap_id_vector>;
using imap_id_count_map = std::unordered_map<imap_id_t, int>;
using imap_id_transform_map = std::map<imap_id_t, imap_id_t>;
using imap_id_transform_map_ptr = std::shared_ptr<imap_id_transform_map>;
using imap_id_uidvalidity_pair = std::pair<imap_id_t, uint64_t>;
using num_t = uint32_t;
using mid_t = uint64_t;
using mid_vector = std::vector<mid_t>;
using mid_vector_ptr = std::shared_ptr<mid_vector>;
using mid_chunks = std::vector<mid_vector>;
using mid_flags_pair = std::pair<mid_t, flags_t>;
using mid_flags_vector = std::vector<mid_flags_pair>;
using mid_flags_vector_ptr = std::shared_ptr<mid_flags_vector>;

using tid_t = uint64_t;
using tid_vector = std::vector<tid_t>;
using tid_vector_ptr = std::shared_ptr<tid_vector>;

using fid_t = std::string;
using fid_vector = std::vector<fid_t>;

namespace time_traits = yplatform::time_traits;

enum class notification_type
{
    normal,
    disabled
};

struct message
{
    enum class status_t : int
    {
        ok = 0,
        to_download_body,
        to_update_flags,
        to_delete,
        to_load_from_local,
    };

    message()
    {
    }

    message(imap_id_t id) : id(id)
    {
    }

    message(imap_id_t id, std::time_t date, flags_t flags) : id(id), date(date), flags(flags)
    {
    }

    message(imap_id_t id, const fid_t& fid, uint32_t errors_count)
        : id(id), fid(fid), errors_count(errors_count)
    {
    }

    num_t num{ 0 };
    imap_id_t id{ 0 };
    std::size_t size{ 0 };
    fid_t fid;
    mid_t mid{ 0 };
    std::time_t date{ std::time(0) };
    flags_t flags;
    status_t status{ status_t::ok };
    uint32_t errors_count{ 0 };
    uint32_t saved_errors_count{ 0 };

    uint32_t get_errors_count_diff()
    {
        return errors_count > saved_errors_count ? errors_count - saved_errors_count : 0;
    }

    bool operator<(const message& other) const
    {
        return id < other.id;
    }

    yplatform::ptree dump();
    std::string to_string(status_t status);
};

using message_opt = boost::optional<message>;
using message_ptr = std::shared_ptr<message>;
using message_vector = std::vector<message>;
using message_vector_ptr = std::shared_ptr<message_vector>;
using imap_id_message_map = std::map<imap_id_t, message>;
using fid_messages_map = std::map<fid_t, message_vector_ptr>;
using message_body_pair = std::pair<message, std::string>;
using message_body_pairs_vector = std::vector<message_body_pair>;

struct move_coordinates
{
    imap_id_t cur_imap_id;
    imap_id_t new_imap_id;
    mid_t mid;
};

using move_coordinates_vec = std::vector<move_coordinates>;

enum class msg_info_type
{
    with_flags,
    without_flags,
};

enum tab_type
{
    relevant,
    news,
    social
};

using tab_opt = std::optional<tab_type>;

namespace message_helpers {

imap_id_vector_ptr to_imap_id_vector(message_vector_ptr messages);

fid_messages_map split_by_fid(message_vector messages);

}
}
