#pragma once

#include "message.h"

#include <common/json.h>

#include <map>
#include <memory>
#include <string>

namespace xeno {

class store_request
{
public:
    using header_map_t = std::map<std::string, std::string>;
    using headers_ptr = std::shared_ptr<header_map_t>;
    using param_map_t = std::multimap<std::string, std::string>;
    using params_ptr = std::shared_ptr<param_map_t>;

    store_request(headers_ptr headers, params_ptr get_params, json::value_ptr post_params)
        : headers_(headers), get_params_(get_params), post_params_(post_params)
    {
    }

    std::string get_value(const std::string& key)
    {
        auto extra_param = extra_params_.find(key);
        if (extra_param != extra_params_.end())
        {
            return extra_param->second;
        }

        auto hdr = headers_->find(key);
        if (hdr != headers_->end())
        {
            return hdr->second;
        }

        auto get_param = get_params_->find(key);
        if (get_param != get_params_->end())
        {
            return get_param->second;
        }

        auto post_param = (*post_params_)[key];
        if (!post_param.isNull() && !post_param.isArray())
        {
            return json::to_string(post_param);
        }

        return std::string();
    }

    void add_value(const std::string& key, const std::string& value)
    {
        extra_params_[key] = value;
    }

    std::vector<std::string> get_string_array(const std::string& key)
    {
        std::vector<std::string> result;
        auto post_param = (*post_params_)[key];
        if (post_param.isArray())
        {
            for (auto& it : post_param)
            {
                result.push_back(json::to_string(it));
            }
        }
        return result;
    }

    std::vector<json::value> get_json_array(const std::string& key)
    {
        std::vector<json::value> result;
        auto post_param = (*post_params_)[key];
        if (post_param.isArray())
        {
            for (auto&& it : post_param)
            {
                result.push_back(it);
            }
        }
        return result;
    }

private:
    headers_ptr headers_;
    params_ptr get_params_;
    json::value_ptr post_params_;
    std::map<std::string, std::string> extra_params_;
};
using store_request_ptr = std::shared_ptr<store_request>;
using send_request = store_request;
using send_request_ptr = std::shared_ptr<send_request>;

}
