#pragma once

#include "folder.h"
#include <xeno/xeno_settings.h>

namespace xeno::mailbox {

enum class turbo_sync_status
{
    required = 0,
    running,
    finished
};

struct sync_newest_state
{
    struct turbo_sync_state
    {
        struct folder_state
        {
            turbo_sync_status status = turbo_sync_status::required;
            time_traits::time_point turbo_sync_start_ts = time_traits::clock::now();
            message_vector_ptr messages_to_download = std::make_shared<message_vector>();
        };

        bool enabled(const mailbox::folder& folder, const turbo_sync_settings& settings)
        {
            auto it = settings.folders.find(folder.type);
            if (it == settings.folders.end())
            {
                return false;
            }
            auto& state = folders[folder.path];
            return (state.status == turbo_sync_status::required ||
                    state.status == turbo_sync_status::running) &&
                time_traits::clock::now() - state.turbo_sync_start_ts < it->second.timeout;
        }

        void enable(const mailbox::folder& folder, const turbo_sync_settings& settings)
        {
            if (!settings.folders.count(folder.type))
            {
                return;
            }
            folders[folder.path] = folder_state();
        }

        std::map<path_t, folder_state> folders;
    };

    struct folder_state
    {
        imap_id_message_map messages_top;
        time_traits::time_point last_sync_ts = time_traits::clock::now();
    };

    time_traits::duration get_total_sync_lag()
    {
        auto now = time_traits::clock::now();
        if (folders.empty())
        {
            return now - creation_ts_;
        }
        auto res = time_traits::duration::min();
        for (auto& [path, state] : folders)
        {
            auto sync_lag = now - state.last_sync_ts;
            res = std::max(res, sync_lag);
        }
        return res;
    }

    time_traits::duration get_folder_sync_lag(path_t path)
    {
        auto last_sync_ts = creation_ts_;
        auto it = folders.find(path);
        if (it != folders.end())
        {
            last_sync_ts = it->second.last_sync_ts;
        }
        return time_traits::clock::now() - last_sync_ts;
    }

    path_t current_folder_path;
    turbo_sync_state turbo_sync;
    std::map<path_t, folder_state> folders;
    time_traits::time_point creation_ts_ = time_traits::clock::now();
};

using sync_newest_state_ptr = std::shared_ptr<sync_newest_state>;
}
