#pragma once

#include "external_mailbox_settings.h"

#include <auth/get_access_token_op.h>
#include <common/types.h>
#include <mailbox/common.h>
#include <ymod_smtpclient/call.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

namespace xeno::mailbox::external {

namespace smtp = ymod_smtpclient;
class check_smtp_credentials_op
    : public std::enable_shared_from_this<check_smtp_credentials_op>
    , public yplatform::log::contains_logger
{
    using client_ptr = boost::shared_ptr<smtp::Call>;
    using proto = smtp::SmtpPoint::Proto;
    using yield_context = yplatform::yield_context<check_smtp_credentials_op>;

public:
    check_smtp_credentials_op(
        context_ptr context,
        client_ptr client,
        const endpoint& smtp_ep,
        const std::string& smtp_login,
        const auth_data& auth,
        const std::string& email,
        settings_ptr settings,
        const without_data_cb& cb,
        const yplatform::log::source& logger)
        : yplatform::log::contains_logger(logger)
        , task_ctx(context)
        , session(client->createSmtpSession(context))
        , smtp_ep(smtp_ep)
        , smtp_login(smtp_login)
        , auth(auth)
        , email(email)
        , settings(settings)
        , cb(cb)
    {
    }

    void operator()(yield_context yield_ctx, error err = {})
    {
        if (err)
        {
            return cb(err);
        }

        try
        {
            reenter(yield_ctx)
            {
                if (auth.type == auth_type::oauth)
                {
                    yield yplatform::spawn(std::make_shared<auth::get_access_token_op>(
                        task_ctx,
                        auth.oauth_app,
                        auth.smtp_credentials,
                        settings->social_settings,
                        false,
                        yield_ctx));
                }

                yield session->asyncConnect(smtp_ep.host, smtp_ep.port, smtp_ep.ssl, yield_ctx);
                yield session->asyncGreeting(yield_ctx);
                yield session->asyncHelo(proto::smtp, settings->hostname_for_smtp, yield_ctx);

                if (!smtp_ep.ssl && session->getServerExtensions().starttls)
                {
                    yield session->asyncStartTls(yield_ctx);
                    yield session->asyncHelo(proto::smtp, settings->hostname_for_smtp, yield_ctx);
                }
                state = operation_state::connected;

                smtp_auth = make_smtp_auth_data(session->getServerExtensions());
                yield session->asyncAuth(smtp_auth, yield_ctx);
                state = operation_state::authenticated;

                yield session->asyncMailFrom(email, yield_ctx);
            }
        }
        catch (const std::exception& e)
        {
            YLOG_L(error) << "check smtp credentials op exception: " << e.what();
            err = code::external_mailbox_exception;
        }

        if (yield_ctx.is_complete())
        {
            cb(err);
        }
    }

    void operator()(yield_context yield_ctx, error err, const std::string& resp, const time_point&)
    {
        if (err)
        {
            YLOG_L(error) << "check smtp credentials op error: can't get access token";
            (*this)(yield_ctx, code::get_access_token_error);
        }
        else
        {
            access_token = resp;
            (*this)(yield_ctx);
        }
    }

    // for asyncConnect, asyncGreeting, asyncStartTls
    void operator()(yield_context yield_ctx, smtp::error::Code err)
    {
        if (err)
        {
            // because now smtpclient does not log error code
            YLOG_L(error) << "check smtp credentials op state=" << to_string(state)
                          << ", error=" << ymod_smtpclient::error::message(err);
            (*this)(yield_ctx, to_error_code(state, ymod_smtpclient::error::message(err)));
        }
        else
        {
            (*this)(yield_ctx);
        }
    }

    // for asyncHelo, asyncAuth, asyncMailFrom
    void operator()(yield_context yield_ctx, smtp::error::Code err, smtp::MultiLineResponse resp)
    {
        if (err)
        {
            auto reason = get_smtp_error_reason(err, resp);
            // because now smtpclient does not log error code
            YLOG_L(error) << "check smtp credentials op state=" << to_string(state)
                          << ", error=" << reason;
            (*this)(yield_ctx, to_error_code(state, reason));
        }
        else
        {
            (*this)(yield_ctx);
        }
    }

private:
    enum class operation_state : int
    {
        not_connected = 0,
        connected,
        authenticated
    };

    error to_error_code(operation_state state, const std::string& reason = "")
    {
        if (state == operation_state::not_connected)
        {
            return error(code::smtp_connect_error, reason);
        }
        else if (state == operation_state::connected)
        {
            return error(code::smtp_auth_error, reason);
        }
        else if (state == operation_state::authenticated)
        {
            return error(code::smtp_check_email_error, reason);
        }
        throw std::runtime_error("bad operation state");
    }

    std::string to_string(operation_state state)
    {
        switch (state)
        {
        case operation_state::not_connected:
            return "not connected";
        case operation_state::connected:
            return "connected";
        case operation_state::authenticated:
            return "authenticated";
        default:
            return "invalid";
        }
    }

    smtp::AuthData make_smtp_auth_data(const smtp::ServerExtensions& extensions)
    {
        if (auth.type == auth_type::oauth)
        {
            return smtp::AuthData::XOAUTH2(smtp_login, access_token);
        }
        else if (extensions.authMechanisms.count(smtp::sasl::Mechanism::Login) > 0)
        {
            return smtp::AuthData::LOGIN(smtp_login, auth.smtp_credentials);
        }
        else if (extensions.authMechanisms.count(smtp::sasl::Mechanism::Plain) > 0)
        {
            return smtp::AuthData::PLAIN(smtp_login, auth.smtp_credentials);
        }
        throw std::runtime_error("not found auth mechanisms");
    }

    std::string get_smtp_error_reason(smtp::error::Code err, const smtp::MultiLineResponse& resp)
    {
        std::stringstream reason;
        reason << ymod_smtpclient::error::message(err) << " reply_code=" << resp.replyCode << " ";
        if (resp.enhancedCode)
        {
            reason << "enhanced_code=" << (*resp.enhancedCode) << " ";
        }
        for (const auto& line : resp.dataLines)
        {
            reason << line << "\\n";
        }
        return reason.str();
    }

    context_ptr task_ctx;
    smtp::SmtpSessionPtr session;
    endpoint smtp_ep;
    std::string smtp_login;
    auth_data auth;
    std::string email;
    settings_ptr settings;
    without_data_cb cb;

    std::string access_token;
    smtp::AuthData smtp_auth;
    operation_state state = operation_state::not_connected;
};

}
