#pragma once

#include "imap_wrapper.h"
#include "external_mailbox_settings.h"

#include <mailbox/data_types/folder.h>
#include <mailbox/common.h>

#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

#include <boost/algorithm/string.hpp>

namespace xeno::mailbox::external {

class delete_messages_op : public yplatform::log::contains_logger
{
    using yield_context = yplatform::yield_context<delete_messages_op>;

public:
    delete_messages_op(
        const path_t& path,
        imap_id_vector_ptr ids,
        const without_data_cb& cb,
        imap_wrapper_ptr client,
        settings_ptr settings,
        const yplatform::log::source& logger)
        : yplatform::log::contains_logger(logger)
        , client(client)
        , path(path.to_string(), path.delim)
        , cb(cb)
        , settings(settings)
    {
        std::vector<std::string> str_ids;
        str_ids.resize(ids->size());
        std::transform(
            ids->begin(),
            ids->end(),
            str_ids.begin(),
            &boost::lexical_cast<std::string, imap_id_t>);
        seq = boost::algorithm::join(str_ids, ",");
    }

    delete_messages_op(
        const path_t& path,
        const std::string& sequence,
        const without_data_cb& cb,
        imap_wrapper_ptr client,
        settings_ptr settings,
        const yplatform::log::source& logger)
        : yplatform::log::contains_logger(logger)
        , client(client)
        , path(path.to_string(), path.delim)
        , seq(sequence)
        , cb(cb)
        , settings(settings)
    {
    }

    void operator()(yield_context ctx, error err = {})
    {
        try
        {
            reenter(ctx)
            {
                yield client->select(path, ctx);
                if (err) yield break;

                yield client->store(seq, "+FLAGS", system_flag_name(system_flag_t::deleted), ctx);
                if (err) yield break;

                yield client->expunge(ctx); // TODO ignore succession of expunge and still reply ok?
                if (err) yield break;
            }
        }
        catch (const std::exception& e)
        {
            YLOG_L(error) << "external mailbox delete_messages_op exception: " << e.what();
            err = code::external_mailbox_exception;
        }

        if (ctx.is_complete())
        {
            cb(err);
        }
    }

private:
    imap_wrapper_ptr client;
    ymod_imap_client::Utf8MailboxName path;
    std::string seq;
    without_data_cb cb;

    settings_ptr settings;
};

}
