#include "external_mailbox_impl.h"
#include "get_folders.h"
#include "get_messages_info_by_id_op.h"
#include "get_messages_info_by_num_op.h"
#include "get_message_body_op.h"
#include "delete_messages_op.h"
#include "move_messages_op.h"
#include "mark_flags_op.h"
#include "send_op.h"
#include "imap_authorize_op.h"
#include "imap_connect_op.h"
#include "check_smtp_credentials_op.h"

namespace xeno::mailbox::external {

void ext_mailbox_impl::get_folder_vector(const folder_vector_cb& cb)
{
    auto handler = [cb, self = shared_from_this(), this](auto err, auto folders) {
        if (err)
        {
            return cb(err, nullptr);
        }
        update_folders_types(folders);
        cb(err, folders);
    };

    try
    {
        external::get_folders(client, settings, handler);
    }
    catch (const std::exception& e)
    {
        YLOG_L(error) << "external mailbox get_folder_vector exception: " << e.what();
        cb(code::external_mailbox_exception, nullptr);
    }
}

void ext_mailbox_impl::get_folder_info(const path_t& path, const folder_cb& cb)
{
    using mailbox_name = ymod_imap_client::Utf8MailboxName;
    using mailbox_result_ptr = ymod_imap_client::ImapMailboxResultPtr;

    auto handler = [path, cb, self = shared_from_this()](
                       error err, mailbox_result_ptr mailbox) {
        folder_ptr result = nullptr;
        if (!err)
        {
            auto& mb = mailbox->mailboxInfo;
            result = std::make_shared<folder>(path, "", mb.uidvalidity_, mb.nextuid_, mb.exists_);
        }
        cb(err, result);
    };

    try
    {
        client->examine_with_counters(
            mailbox_name(path.to_string(), path.delim), std::move(handler));
    }
    catch (const std::exception& e)
    {
        YLOG_L(error) << "external mailbox get_folder_info exception: " << e.what();
        cb(code::external_mailbox_exception, nullptr);
    }
}

void ext_mailbox_impl::get_messages_info_by_id(
    const path_t& path,
    const imap_range& range,
    const messages_vector_cb& cb)
{
    spawn<get_messages_info_by_id_op>(ext_ctx, path, range, cb);
}

void ext_mailbox_impl::get_messages_info_by_num(
    const path_t& path,
    num_t top,
    num_t bottom,
    const messages_vector_cb& cb)
{
    spawn<get_messages_info_by_num_op>(ext_ctx, path, top, bottom, cb);
}

void ext_mailbox_impl::get_message_body(const path_t& path, imap_id_t id, const message_body_cb& cb)
{
    spawn<get_message_body_op>(path, id, cb);
}

void ext_mailbox_impl::delete_messages(
    const path_t& path,
    imap_id_vector_ptr ids,
    const without_data_cb& cb)
{
    spawn<delete_messages_op>(path, ids, cb);
}

void ext_mailbox_impl::move_messages(
    const path_t& from,
    const path_t& to,
    imap_id_vector_ptr ids,
    const imap_ids_transform_cb& cb)
{
    spawn<move_messages_op>(from, to, ids, cb);
}

void ext_mailbox_impl::move_all_messages(
    const path_t& from,
    const path_t& to,
    const imap_ids_transform_cb& cb)
{
    spawn<move_messages_op>(from, to, "1:*", cb);
}

void ext_mailbox_impl::create_folder(const path_t& path, const without_data_cb& cb)
{
    try
    {
        client->create_mailbox(ymod_imap_client::Utf8MailboxName(path.to_string(), path.delim), cb);
    }
    catch (std::exception& e)
    {
        YLOG_L(error) << "external mailbox create_folder exception: " << e.what();
        cb(code::external_mailbox_exception);
    }
}

void ext_mailbox_impl::rename_folder(
    const path_t& old,
    const path_t& new_,
    const without_data_cb& cb)
{
    try
    {
        client->rename_mailbox(
            ymod_imap_client::Utf8MailboxName(old.to_string(), old.delim),
            ymod_imap_client::Utf8MailboxName(new_.to_string(), new_.delim),
            cb);
    }
    catch (std::exception& e)
    {
        YLOG_L(error) << "external mailbox rename_folder exception: " << e.what();
        cb(code::external_mailbox_exception);
    }
}

void ext_mailbox_impl::delete_folder(const path_t& path, const without_data_cb& cb)
{
    try
    {
        client->delete_mailbox(ymod_imap_client::Utf8MailboxName(path.to_string(), path.delim), cb);
    }
    catch (std::exception& e)
    {
        YLOG_L(error) << "external mailbox delete_folder exception: " << e.what();
        cb(code::external_mailbox_exception);
    }
}

void ext_mailbox_impl::clear_folder(const path_t& path, const without_data_cb& cb)
{
    spawn<delete_messages_op>(path, "1:*", cb);
}

void ext_mailbox_impl::mark_flags(
    const path_t& path,
    imap_id_vector_ptr ids,
    const flags_t& add,
    const flags_t& del,
    const without_data_cb& cb)
{
    spawn<mark_flags_op>(path, ids, add, del, cb);
}

void ext_mailbox_impl::append(
    const path_t& path,
    std::string&& body,
    const flags_t& flags,
    const std::string& date,
    const imap_id_and_uidvalidity_cb& cb)
{
    try
    {
        std::string result_flags;
        yplatform::sstream flags_stream(result_flags);

        std::vector<std::string> system_flags;
        for (auto& flag : flags.system_flags)
        {
            system_flags.push_back(system_flag_name(flag));
        }
        flags_stream << boost::algorithm::join(system_flags, " ")
                     << boost::algorithm::join(flags.user_flags, " ");

        client->append(
            std::move(body),
            ymod_imap_client::Utf8MailboxName(path.to_string(), path.delim),
            result_flags,
            date,
            [cb, self = shared_from_this()](
                error err, ymod_imap_client::AppenduidResultPtr append_result) {
                imap_id_uidvalidity_pair result;
                if (!err)
                {
                    result.first = append_result->uid;
                    result.second = append_result->uidvalidity;
                }
                cb(err, std::move(result));
            });
    }
    catch (std::exception& e)
    {
        YLOG_L(error) << "external mailbox append exception: " << e.what();
        cb(code::external_mailbox_exception, imap_id_uidvalidity_pair());
    }
}

void ext_mailbox_impl::send(
    const std::string& smtp_login,
    const auth_data& data,
    const endpoint& smtp_ep,
    const std::string& from,
    const std::vector<std::string>& to,
    std::shared_ptr<std::string> body,
    bool notify,
    const without_data_cb& cb)
{
    auto smtp_client = yplatform::find<ymod_smtpclient::Call>("smtp_client");
    auto coro = std::make_shared<send_op>(
        smtp_client, context, settings, smtp_login, data, smtp_ep, from, to, body, notify, cb);
    yplatform::spawn(coro);
}

void ext_mailbox_impl::imap_authorize(
    const std::string& imap_login,
    const auth_data& data,
    const endpoint& imap_ep,
    const without_data_cb& cb)
{
    imap_connect(
        imap_ep, [imap_login, data, imap_ep, cb, self = shared_from_this(), this](error ec) {
            if (ec)
            {
                reset();
                return cb(ec);
            }
            spawn<imap_authorize_op>(context, imap_login, data, imap_ep, cb);
        });
}

void ext_mailbox_impl::get_provider(const endpoint& imap_ep, const provider_cb& cb)
{
    imap_connect(imap_ep, [cb, self = shared_from_this(), this](error ec) {
        std::string provider;
        if (!ec)
        {
            provider = client->get_provider();
        }
        else
        {
            reset();
        }
        cb(ec, provider);
    });
}

std::string ext_mailbox_impl::get_provider_unsafe()
{
    return client ? client->get_provider() : "custom";
}

void ext_mailbox_impl::imap_connect(const endpoint& imap_ep, const without_data_cb& cb)
{
    bool connected = client ? client->connected() : false;
    if (connected)
    {
        return cb(error());
    }

    if (!client)
    {
        client = std::make_shared<imap_wrapper>(io, context, settings->imap_wrapper_settings);
    }
    spawn<imap_connect_op>(context, imap_ep, cb, io);
}

void ext_mailbox_impl::check_smtp_credentials(
    const std::string& smtp_login,
    const auth_data& data,
    const endpoint& smtp_ep,
    const std::string& email,
    const without_data_cb& cb)
{
    auto smtp_client = yplatform::find<ymod_smtpclient::Call>("smtp_client");
    auto coro = std::make_shared<check_smtp_credentials_op>(
        context, smtp_client, smtp_ep, smtp_login, data, email, settings, cb, logger());
    yplatform::spawn(coro);
}

bool ext_mailbox_impl::authenticated()
{
    return client && client->authenticated();
}

void ext_mailbox_impl::reset()
{
    if (client)
    {
        saved_stats += client->get_stats();
        client->reset();
    }
}

statistics ext_mailbox_impl::get_stats()
{
    auto stats = client ? client->get_stats() : statistics();
    stats += saved_stats;
    return stats;
}

void ext_mailbox_impl::update_folders_types(folder_vector_ptr folders)
{
    for (auto cache_it = ext_ctx->folders_types.begin(); cache_it != ext_ctx->folders_types.end();)
    {
        auto folder_it = std::find_if(
            folders->begin(), folders->end(), [path = cache_it->first](const folder& folder) {
                return path == folder.path.to_string();
            });
        if (folder_it == folders->end())
        {
            cache_it = ext_ctx->folders_types.erase(cache_it);
        }
        else
        {
            ++cache_it;
        }
    }

    for (auto& folder : *folders)
    {
        ext_ctx->folders_types[folder.path.to_string()] = folder.type;
    }
}

}
