#pragma once

#include "external_mailbox_settings.h"

#include "imap_wrapper.h"

#include <mailbox/common.h>
#include <mailbox/external/statistics.h>
#include <mailbox/data_types/folder.h>

#include <yplatform/task_context.h>
#include <yplatform/find.h>

namespace xeno::mailbox::external {

class ext_mailbox_impl final
    : public std::enable_shared_from_this<ext_mailbox_impl>
    , public yplatform::log::contains_logger
{
public:
    ext_mailbox_impl(
        boost::asio::io_service& io,
        yplatform::task_context_ptr context,
        settings_ptr settings,
        const yplatform::log::source& logger)
        : yplatform::log::contains_logger(logger), io(io), context(context), settings(settings)
    {
    }

    void imap_authorize(
        const std::string& imap_login,
        const auth_data& data,
        const endpoint& imap_ep,
        const without_data_cb& cb);

    void get_provider(const endpoint& imap_ep, const provider_cb& cb);

    // This method is not thread-safe. Don't call it in parallel with any other methods.
    std::string get_provider_unsafe();

    void check_smtp_credentials(
        const std::string& smtp_login,
        const auth_data& data,
        const endpoint& smtp_ep,
        const std::string& email,
        const without_data_cb& cb);

    void get_folder_vector(const folder_vector_cb& cb);
    void get_folder_info(const path_t& path, const folder_cb& cb);

    void get_messages_info_by_id(
        const path_t& path,
        const imap_range& range,
        const messages_vector_cb& cb);
    void get_messages_info_by_num(
        const path_t& path,
        num_t top,
        num_t bottom,
        const messages_vector_cb& cb);
    void get_message_body(const path_t& path, imap_id_t id, const message_body_cb& cb);

    void delete_messages(const path_t& path, imap_id_vector_ptr ids, const without_data_cb& cb);
    void move_messages(
        const path_t& from,
        const path_t& to,
        imap_id_vector_ptr ids,
        const imap_ids_transform_cb& cb);
    void move_all_messages(const path_t& from, const path_t& to, const imap_ids_transform_cb& cb);

    void create_folder(const path_t& path, const without_data_cb& cb);
    void rename_folder(const path_t& old, const path_t& new_, const without_data_cb& cb);
    void delete_folder(const path_t& path, const without_data_cb& cb);
    void clear_folder(const path_t& path, const without_data_cb& cb);

    void mark_flags(
        const path_t& path,
        imap_id_vector_ptr ids,
        const flags_t& add,
        const flags_t& del,
        const without_data_cb& cb);

    void append(
        const path_t& path,
        std::string&& body,
        const flags_t& flags,
        const std::string& date,
        const imap_id_and_uidvalidity_cb& cb);
    void send(
        const std::string& smtp_login,
        const auth_data& data,
        const endpoint& smtp_ep,
        const std::string& from,
        const std::vector<std::string>& to,
        std::shared_ptr<std::string> body,
        bool notify,
        const without_data_cb& cb);

    bool authenticated();

    void reset();

    statistics get_stats();

private:
    template <typename Coroutine, typename... Args>
    void spawn(Args&&... args)
    {
        auto coro =
            std::make_shared<Coroutine>(std::forward<Args>(args)..., client, settings, logger());
        yplatform::spawn(coro);
    }

    void imap_connect(const endpoint& imap_ep, const without_data_cb& cb);
    void update_folders_types(folder_vector_ptr folders);

    boost::asio::io_service& io;
    yplatform::task_context_ptr context;
    settings_ptr settings;
    statistics saved_stats;
    imap_wrapper_ptr client;
    ymod_imap_client::ImapCapabilityPtr capabilities;
    ext_mailbox_ctx_ptr ext_ctx = std::make_shared<ext_mailbox_ctx>();
};

}
