#pragma once

#include "ptree_converters.h"
#include <auth/social/settings.h>
#include <common/errors.h>
#include <common/ip.h>
#include <mailbox/data_types/folder.h>
#include <mailbox/external/imap_wrapper.h>

#include <ymod_imapclient/imap_client.h>
#include <yplatform/ptree.h>
#include <boost/range/iterator_range.hpp>
#include <boost/algorithm/string.hpp>

namespace xeno::mailbox::external {

template <typename DstMap, typename SrcMap>
DstMap inverse_map(const SrcMap& src)
{
    DstMap res;
    for (auto&& [key, val] : src)
    {
        res[val] = key;
    }
    return res;
}

struct settings
{
    void update(const yplatform::ptree& conf)
    {
        std::multimap<folder::type_t, string> type_names;
        yplatform::read_ptree(type_names, conf, "folder_types");
        for (auto&& [type, name] : type_names)
        {
            folders_types[boost::algorithm::to_lower_copy(name)] = type;
        }

        social_settings = std::make_shared<auth::social::social_settings>(conf.get_child("social"));
        yplatform::read_ptree(oauth_login_type, conf, "oauth_app_login_types");

        auto wrapper_st = std::make_shared<imap_wrapper::settings>();
        auto responses_cfg = conf.get_child("imap_auth_responses");
        auto protocol_disabled = responses_cfg.equal_range("protocol_disabled");
        for (auto& [name, entry] : boost::make_iterator_range(protocol_disabled))
        {
            wrapper_st->auth_responses[entry.data()] = code::imap_protocol_disabled;
        }
        auto account_blocked = responses_cfg.equal_range("account_blocked");
        for (auto& [name, entry] : boost::make_iterator_range(account_blocked))
        {
            wrapper_st->auth_responses[entry.data()] = code::imap_account_blocked;
        }

        for (auto&& [provider, provider_conf] : conf.get_child("provider_detection"))
        {
            auto id_name_response = provider_conf.get_optional<string>("id_name_response");
            if (id_name_response)
            {
                wrapper_st->id_responses[*id_name_response] = provider;
            }

            std::vector<string> servers;
            yplatform::read_ptree(servers, provider_conf, "servers");
            for (auto&& server : servers)
            {
                wrapper_st->provider_by_server[server] = provider;
            }
        }
        yplatform::read_ptree(wrapper_st->command_timeouts, conf, "imap_command_timeouts");
        imap_wrapper_settings = wrapper_st;

        hostname_for_smtp = conf.get<std::string>("hostname_for_smtp", "");

        std::vector<std::string> networks;
        yplatform::read_ptree(networks, conf, "blacklisted_networks");
        std::transform(
            networks.begin(),
            networks.end(),
            std::back_inserter(blacklisted_networks),
            &make_ip_network);
    }

    std::map<std::string, folder::type_t> folders_types;

    auth::social::social_settings_ptr social_settings;
    imap_wrapper::settings_ptr imap_wrapper_settings;
    std::map<std::string, ymod_imap_client::ImapClient::OauthLoginType> oauth_login_type;

    std::string hostname_for_smtp;

    std::vector<ip_network> blacklisted_networks;
};

using settings_ptr = std::shared_ptr<settings>;

}
