#include "get_folders.h"

namespace xeno::mailbox::external {

using namespace ymod_imap_client;
using type_set = std::set<folder::type_t>;
using imap_list_item_comparator = std::function<bool(ImapListItemPtr, ImapListItemPtr)>;
using folder_ptr_set = std::set<ImapListItemPtr, imap_list_item_comparator>;

auto gmail_folder_name = "[Gmail]";

bool compare_imap_list_items(ImapListItemPtr lhs, ImapListItemPtr rhs)
{
    return lhs->name.asString() < rhs->name.asString();
}

folder::type_t get_type_from_flags(uint32_t flags)
{
    auto type = folder::type_t::user;
    if (flags & ListResponse::ff_sent)
    {
        type = folder::type_t::sent;
    }
    else if (flags & ListResponse::ff_trash)
    {
        type = folder::type_t::trash;
    }
    else if (flags & ListResponse::ff_spam)
    {
        type = folder::type_t::spam;
    }
    else if (flags & ListResponse::ff_drafts)
    {
        type = folder::type_t::drafts;
    }
    return type;
}

folder::type_t get_type_from_name(const std::string& name, const folder_name_type_map& folders_type)
{
    auto type_it = folders_type.find(boost::algorithm::to_lower_copy(name));
    return (type_it == folders_type.end() ? folder::type_t::user : type_it->second);
}

bool is_system_folder(ImapListItemPtr folder)
{
    return folder->flags & ListResponse::ff_sent || folder->flags & ListResponse::ff_trash ||
        folder->flags & ListResponse::ff_spam || folder->flags & ListResponse::ff_drafts;
}

bool is_special_folder(ImapListItemPtr folder)
{
    return folder->flags & ListResponse::ff_all || folder->flags & ListResponse::ff_important ||
        folder->flags & ListResponse::ff_starred;
}

void set_type(folder& folder, folder::type_t type, type_set& taken_types)
{
    if (taken_types.find(type) == taken_types.end())
    {
        taken_types.insert(type);
        folder.type = type;
    }
}

void add_folder_with_missed_parents(
    ImapListItemPtr external_folder,
    folder_ptr_set& result,
    bool is_gmail)
{
    auto delim = external_folder->name.getDelimiter();
    auto folder_name = external_folder->name.asString();
    auto end_of_processing_path = std::find(folder_name.begin(), folder_name.end(), delim);
    // find first missed parent
    while (end_of_processing_path != folder_name.end())
    {
        std::string parent_name = { folder_name.begin(), end_of_processing_path };
        bool is_gmail_child = is_gmail && parent_name == gmail_folder_name;
        auto mailbox_ptr = std::make_shared<ImapListItem>(Utf8MailboxName(parent_name, delim), 0);

        if (result.find(mailbox_ptr) != result.end() ||
            (is_system_folder(external_folder) && is_gmail_child))
        {
            end_of_processing_path =
                std::find(end_of_processing_path + 1, folder_name.end(), delim);
            continue;
        }
        break;
    }

    // add each parent since first missed
    while (end_of_processing_path != folder_name.end())
    {
        std::string parent_name = { folder_name.begin(), end_of_processing_path };
        result.insert(std::make_shared<ImapListItem>(Utf8MailboxName(parent_name, delim), 0));
        end_of_processing_path = std::find(end_of_processing_path + 1, folder_name.end(), delim);
    }
    result.insert(external_folder);
}

// sorting mailboxes by name need to guarantee same order every sync
ImapListPtr fix_folders_hierarchy(ImapListPtr imap_list, bool is_gmail)
{
    folder_ptr_set result(compare_imap_list_items);
    std::sort(imap_list->mailboxes.begin(), imap_list->mailboxes.end(), compare_imap_list_items);

    for (auto folder : imap_list->mailboxes)
    {
        if (is_special_folder(folder))
        {
            continue;
        }
        if ((folder->flags & ListResponse::ff_noselect) && is_gmail &&
            folder->name.asString() == gmail_folder_name)
        {
            continue;
        }
        add_folder_with_missed_parents(folder, result, is_gmail);
    }
    return std::make_shared<ImapList>(ImapMailboxList(result.begin(), result.end()));
}

folder_vector_ptr convert_imap_folders(
    ImapListPtr imap_list,
    const folder_name_type_map& folders_type)
{
    auto result = std::make_shared<folder_vector>();
    type_set taken_types;
    for (auto folder : imap_list->mailboxes)
    {
        result->emplace_back(
            path_t{ folder->name.asString(), folder->name.getDelimiter() }, "", 0, 0, 0);
        set_type(result->back(), get_type_from_flags(folder->flags), taken_types);
    }
    // set types from name for system folders in result
    for (auto& folder : *result)
    {
        if (folder.type == folder::type_t::user)
        {
            set_type(
                folder, get_type_from_name(folder.path.to_string(), folders_type), taken_types);
        }
    }
    return result;
}

void get_folders(imap_wrapper_ptr client, settings_ptr settings, const folder_vector_cb& cb)
{
    auto handler = [client, settings, cb](auto err, auto imap_folders) {
        if (err)
        {
            return cb(err, nullptr);
        }
        imap_folders = fix_folders_hierarchy(imap_folders, client->get_capability().gmailEx);
        auto folders = convert_imap_folders(imap_folders, settings->folders_types);
        cb(err, folders);
    };

    client->list(std::move(handler));
}

}
