#pragma once

#include "imap_wrapper.h"
#include "external_mailbox_settings.h"

#include <mailbox/data_types/folder.h>
#include <mailbox/common.h>

#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

namespace xeno::mailbox::external {

class get_message_body_op : public yplatform::log::contains_logger
{
    using yield_context = yplatform::yield_context<get_message_body_op>;

public:
    get_message_body_op(
        const path_t& path,
        imap_id_t id,
        const message_body_cb& cb,
        imap_wrapper_ptr client,
        settings_ptr settings,
        const yplatform::log::source& logger)
        : yplatform::log::contains_logger(logger)
        , client(client)
        , path(path.to_string(), path.delim)
        , id(id)
        , cb(cb)
        , settings(settings)
    {
    }

    void operator()(yield_context ctx, error err = {})
    {
        try
        {
            reenter(ctx)
            {
                yield client->examine(path, ctx);
                if (err) yield break;

                yield client->uid_fetch_body(boost::lexical_cast<std::string>(id), ctx);
                if (err) yield break;

                if (imapResult->messages.empty())
                {
                    err = code::message_not_found;
                    yield break;
                }

                result =
                    std::make_shared<std::string>(std::move(imapResult->messages.front().body));
            }
        }
        catch (const std::exception& e)
        {
            YLOG_L(error) << "external mailbox get_message_body_op exception: " << e.what();
            err = code::external_mailbox_exception;
        }

        if (ctx.is_complete())
        {
            cb(err, result);
        }
    }

    void operator()(yield_context ctx, error err, ymod_imap_client::MessageSetPtr messageResult)
    {
        imapResult = messageResult;
        (*this)(ctx, err);
    }

private:
    imap_wrapper_ptr client;

    ymod_imap_client::Utf8MailboxName path;
    imap_id_t id;

    ymod_imap_client::MessageSetPtr imapResult;

    message_body_cb cb;
    string_ptr result;

    settings_ptr settings;
};

}
