#pragma once

#include "imap_wrapper.h"
#include "external_mailbox_settings.h"

#include <auth/get_access_token_op.h>
#include <mailbox/common.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

namespace xeno::mailbox::external {

class imap_authorize_op : public yplatform::log::contains_logger
{
    using yield_context = yplatform::yield_context<imap_authorize_op>;

public:
    imap_authorize_op(
        yplatform::task_context_ptr context,
        const std::string& imap_login,
        const auth_data& data,
        const endpoint& imap_ep,
        const without_data_cb& cb,
        imap_wrapper_ptr client,
        settings_ptr settings,
        const yplatform::log::source& logger)
        : yplatform::log::contains_logger(logger)
        , client(client)
        , context(context)
        , settings(settings)
        , imap_login(imap_login)
        , data(data)
        , imap_ep(imap_ep)
        , cb(cb)
    {
    }

    void operator()(yield_context ctx, error err = {})
    {
        try
        {
            reenter(ctx)
            {
                if (data.type == auth_type::password)
                {
                    if (client->get_capability().authPlain)
                    {
                        yield client->auth_plain(imap_login, data.imap_credentials, ctx);
                    }
                    else
                    {
                        yield client->login(imap_login, data.imap_credentials, ctx);
                    }
                }
                else
                {
                    yield yplatform::spawn(std::make_shared<auth::get_access_token_op>(
                        context,
                        data.oauth_app,
                        data.imap_credentials,
                        settings->social_settings,
                        false,
                        ctx));
                    if (!err)
                    {
                        type_it = settings->oauth_login_type.find(data.oauth_app);
                        if (type_it != settings->oauth_login_type.end())
                        {
                            login_type = type_it->second;
                        }

                        yield client->login_oauth(imap_login, access_token, login_type, ctx);
                    }
                }

                if (err)
                {
                    client->reset();
                    yield break;
                }

                yield client->load_capability(ctx);
                if (err)
                {
                    if (err == errc::imap_not_connected)
                    {
                        yield break;
                    }
                    err = code::ok;
                }
            }
        }
        catch (const std::exception& e)
        {
            YLOG_L(error) << "external mailbox imap_authorize_op exception: " << e.what();
            err = code::external_mailbox_exception;
        }

        if (ctx.is_complete())
        {
            cb(err);
        }
    }

    void operator()(yield_context ctx, error err, const std::string& resp, const time_point&)
    {
        access_token = resp;
        (*this)(ctx, err);
    }

    void operator()(yield_context ctx, error err, ymod_imap_client::ConnectResultPtr)
    {
        (*this)(ctx, err);
    }

private:
    imap_wrapper_ptr client;
    yplatform::task_context_ptr context;
    settings_ptr settings;
    std::string imap_login;
    auth_data data;
    endpoint imap_ep;
    without_data_cb cb;

    std::string access_token;

    ymod_imap_client::ImapClient::OauthLoginType login_type =
        ymod_imap_client::ImapClient::OauthLoginType::Multiline;
    std::map<std::string, ymod_imap_client::ImapClient::OauthLoginType>::iterator type_it;
};

}
