#pragma once

#include "imap_wrapper.h"
#include "external_mailbox_settings.h"
#include "resolve_server_op.h"

#include <mailbox/common.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

namespace xeno::mailbox::external {

class imap_connect_op : public yplatform::log::contains_logger
{
    using yield_context = yplatform::yield_context<imap_connect_op>;

public:
    imap_connect_op(
        yplatform::task_context_ptr context,
        const endpoint& imap_ep,
        const without_data_cb& cb,
        boost::asio::io_service& io,
        imap_wrapper_ptr client,
        settings_ptr settings,
        const yplatform::log::source& logger)
        : yplatform::log::contains_logger(logger)
        , client(client)
        , context(context)
        , settings(settings)
        , imap_ep(imap_ep)
        , cb(cb)
        , io(io)
    {
    }

    void operator()(yield_context ctx, error err = {}, const std::string& resolved = "")
    {
        try
        {
            reenter(ctx)
            {
                yield yplatform::spawn(
                    std::make_shared<resolve_server_op>(imap_ep.host, io, settings, logger(), ctx));
                if (err) yield break;

                yield client->connect(resolved, imap_ep.host, imap_ep.port, imap_ep.ssl, ctx);
                if (err)
                {
                    err = code::imap_connect_error;
                    yield break;
                }

                yield client->load_capability(ctx);
                if (err)
                {
                    if (err == errc::imap_not_connected)
                    {
                        yield break;
                    }
                    err = code::ok;
                }

                if (client->get_capability().id)
                {
                    yield client->id(ctx);
                    if (err)
                    {
                        if (err == errc::imap_not_connected)
                        {
                            yield break;
                        }
                        err = code::ok;
                    }
                }

                if (!imap_ep.ssl)
                {
                    yield client->start_tls(ctx);
                    if (err)
                    {
                        client->reset();
                        yield break;
                    }
                }
            }
        }
        catch (const std::exception& e)
        {
            YLOG_L(error) << "external mailbox imap_connect_op exception: " << e.what();
            err = code::external_mailbox_exception;
        }

        if (ctx.is_complete())
        {
            cb(err);
        }
    }

    void operator()(yield_context ctx, error err, ymod_imap_client::ConnectResultPtr)
    {
        (*this)(ctx, err);
    }

private:
    imap_wrapper_ptr client;
    yplatform::task_context_ptr context;
    settings_ptr settings;
    endpoint imap_ep;
    without_data_cb cb;
    boost::asio::io_service& io;
};

}
