#pragma once

#include "statistics.h"

#include <common/errors.h>
#include <common/types.h>

#include <ymod_imapclient/call.h>
#include <ymod_imapclient/errors.h>

#include <yplatform/find.h>

#include <boost/asio/io_service.hpp>
#include <boost/asio/strand.hpp>

#include <map>
#include <memory>

namespace xeno::mailbox::external {

template <typename Result>
struct imap_call_wrapper : public yplatform::log::contains_logger
{
    imap_call_wrapper(ymod_imap_client::ImapClientPtr& client) : client(client)
    {
    }

    template <typename Method, typename Handler, typename... Args>
    void operator()(Handler&& h, Method m, Args&&... args)
    {
        auto future = ((*client).*m)(std::forward<Args>(args)...);
        future.add_callback([handler = std::move(h), future, logger = logger()]() mutable {
            try
            {
                auto res = future.get();
                handler(code::ok, "", std::move(res));
            }
            catch (const ymod_imap_client::NoException& e)
            {
                YLOG(logger, error) << "imap response no: " << e.what();
                handler(error(code::imap_response_no, e.what()), e.reason, Result());
            }
            catch (const ymod_imap_client::BadException& e)
            {
                YLOG(logger, error) << "imap response bad: " << e.what();
                handler(error(code::imap_response_bad, e.what()), "", Result());
            }
            catch (const ymod_imap_client::TimeoutException& e)
            {
                YLOG(logger, error) << "imap timeout: " << e.what();
                handler(code::imap_timeout, "", Result());
            }
            catch (const std::exception& e)
            {
                YLOG(logger, error) << "imap error: " << e.what();
                handler(error(code::imap_general_error, e.what()), "", Result());
            }
        });
    }

    ymod_imap_client::ImapClientPtr& client;
};

template <>
struct imap_call_wrapper<ymod_imap_client::ImapResultPtr> : public yplatform::log::contains_logger
{
    imap_call_wrapper(ymod_imap_client::ImapClientPtr& client) : client(client)
    {
    }

    template <typename Method, typename Handler, typename... Args>
    void operator()(Handler&& h, Method m, Args&&... args)
    {
        auto future = ((*client).*m)(std::forward<Args>(args)...);
        future.add_callback([handler = std::move(h), future, logger = logger()]() mutable {
            try
            {
                auto res = future.get();
                handler(code::ok, "");
            }
            catch (const ymod_imap_client::NoException& e)
            {
                YLOG(logger, error) << "imap response no: " << e.what();
                handler(error(code::imap_response_no, e.what()), e.reason);
            }
            catch (const ymod_imap_client::BadException& e)
            {
                YLOG(logger, error) << "imap response bad: " << e.what();
                handler(error(code::imap_response_bad, e.what()), "");
            }
            catch (const ymod_imap_client::TimeoutException& e)
            {
                YLOG(logger, error) << "imap timeout: " << e.what();
                handler(code::imap_timeout, "");
            }
            catch (const std::exception& e)
            {
                YLOG(logger, error) << "imap error: " << e.what();
                handler(error(code::imap_general_error, e.what()), "");
            }
        });
    }

    ymod_imap_client::ImapClientPtr& client;
};

using auth_responses = std::map<std::string, code>;
using id_name_provider_responses = std::map<std::string, std::string>;
using imap_server_to_provider_map = std::map<std::string, std::string>;
using command_timeouts = std::map<std::string, time_traits::duration>;

class imap_wrapper : public std::enable_shared_from_this<imap_wrapper>
{
    using connect_method_ptr =
        ymod_imap_client::FutureConnectResult (ymod_imap_client::ImapClient::*)(
            const std::string& server,
            const std::string& server_fqdn,
            unsigned short port,
            bool ssl);

public:
    using oauth_type_t = ymod_imap_client::ImapClient::OauthLoginType;

    struct settings
    {
        auth_responses auth_responses;
        id_name_provider_responses id_responses;
        command_timeouts command_timeouts;
        imap_server_to_provider_map provider_by_server;
    };
    using settings_ptr = std::shared_ptr<settings>;

    enum class state_t : int
    {
        // Don't change order
        not_connected = 0,
        connected,
        authenticated,
        examined,
        selected
    };

    imap_wrapper(boost::asio::io_service& io, context_ptr context, settings_ptr settings)
        : strand(io), context(context), st(settings)
    {
        auto module = yplatform::find<ymod_imap_client::call>("imap_client");
        client = module->clientInstance(context);
    }

    template <typename Handler>
    void connect(
        const std::string& server,
        const std::string& server_fqdn,
        unsigned short port,
        bool ssl,
        Handler h)
    {
        client->setCommandTimeout(get_timeout("connect"));
        imap_call_wrapper<ymod_imap_client::ConnectResultPtr> wrapper(client);
        wrapper.logger(context->logger());
        auto cb = [this, self = shared_from_this(), h](
                      error err,
                      const std::string&,
                      ymod_imap_client::ConnectResultPtr&& result) mutable {
            handle_status(err);
            state = err ? state_t::not_connected : state_t::connected;
            h(err, result);
        };

        provider = get_provider_by_server(server_fqdn);
        wrapper(
            std::move(cb),
            connect_method_ptr(&ymod_imap_client::ImapClient::connect),
            server,
            server_fqdn,
            port,
            ssl);
    }

    template <typename Handler>
    void start_tls(Handler h)
    {
        client->setCommandTimeout(get_timeout("start_tls"));
        imap_call_wrapper<ymod_imap_client::ImapResultPtr> wrapper(client);
        wrapper.logger(context->logger());
        auto cb = [this, self = shared_from_this(), h](error err, const std::string&) mutable {
            handle_status(err);
            h(err);
        };
        wrapper(std::move(cb), &ymod_imap_client::ImapClient::startTls);
    }

    template <typename Handler>
    void login(const std::string& user, const std::string& password, Handler h)
    {
        client->setCommandTimeout(get_timeout("login"));
        imap_call_wrapper<ymod_imap_client::ImapResultPtr> wrapper(client);
        wrapper.logger(context->logger());
        auto cb =
            [this, self = shared_from_this(), h](error err, const std::string& reason) mutable {
                handle_status(err);
                if (err)
                {
                    clarify_error_with_reason(err, reason);
                }
                else
                {
                    state = state_t::authenticated;
                }
                h(err);
            };
        wrapper(std::move(cb), &ymod_imap_client::ImapClient::login, user, password);
    }

    template <typename Handler>
    void auth_plain(const std::string& user, const std::string& password, Handler h)
    {
        client->setCommandTimeout(get_timeout("auth_plain"));
        imap_call_wrapper<ymod_imap_client::ImapResultPtr> wrapper(client);
        wrapper.logger(context->logger());
        auto cb =
            [this, self = shared_from_this(), h](error err, const std::string& reason) mutable {
                handle_status(err);
                if (err)
                {
                    clarify_error_with_reason(err, reason);
                }
                else
                {
                    state = state_t::authenticated;
                }
                h(err);
            };
        wrapper(std::move(cb), &ymod_imap_client::ImapClient::authPlain, user, password);
    }

    template <typename Handler>
    void login_oauth(
        const std::string& user,
        const std::string& token,
        oauth_type_t type,
        Handler h)
    {
        client->setCommandTimeout(get_timeout("login_oauth"));
        imap_call_wrapper<ymod_imap_client::ImapResultPtr> wrapper(client);
        wrapper.logger(context->logger());
        auto cb = [this, self = shared_from_this(), h](error err, const std::string&) mutable {
            handle_status(err);
            if (!err)
            {
                state = state_t::authenticated;
            }
            h(err);
        };
        wrapper(std::move(cb), &ymod_imap_client::ImapClient::loginOauth, user, token, type);
    }

    template <typename Handler>
    void load_capability(Handler h)
    {
        client->setCommandTimeout(get_timeout("load_capability"));
        imap_call_wrapper<ymod_imap_client::ImapCapabilityPtr> wrapper(client);
        wrapper.logger(context->logger());
        auto cb = [this, self = shared_from_this(), h](
                      error err,
                      const std::string&,
                      ymod_imap_client::ImapCapabilityPtr&& result) mutable {
            handle_status(err);
            if (!err)
            {
                handle_capability(result);
            }
            h(err);
        };
        wrapper(std::move(cb), &ymod_imap_client::ImapClient::capability);
    }

    template <typename Handler>
    void id(Handler h)
    {
        client->setCommandTimeout(get_timeout("id"));
        imap_call_wrapper<ymod_imap_client::IdResultPtr> wrapper(client);
        wrapper.logger(context->logger());
        auto cb =
            [this, self = shared_from_this(), h](
                error err, const std::string&, ymod_imap_client::IdResultPtr&& result) mutable {
                handle_status(err);
                if (!err)
                {
                    handle_id(result);
                }
                h(err);
            };
        wrapper(std::move(cb), &ymod_imap_client::ImapClient::id, "");
    }

    template <typename Handler>
    void list(Handler h)
    {
        client->setCommandTimeout(get_timeout("list"));
        imap_call_wrapper<ymod_imap_client::ImapListPtr> wrapper(client);
        wrapper.logger(context->logger());
        auto cb =
            [this, self = shared_from_this(), h](
                error err, const std::string&, ymod_imap_client::ImapListPtr&& result) mutable {
                handle_status(err);
                h(err, result);
            };
        wrapper(std::move(cb), &ymod_imap_client::ImapClient::list, "", "*");
    }

    template <typename Handler>
    void status(const ymod_imap_client::Utf8MailboxName& name, Handler h)
    {
        client->setCommandTimeout(get_timeout("status"));
        imap_call_wrapper<ymod_imap_client::ImapStatusPtr> wrapper(client);
        wrapper.logger(context->logger());
        auto cb =
            [this, self = shared_from_this(), h](
                error err, const std::string&, ymod_imap_client::ImapStatusPtr&& result) mutable {
                handle_status(err);
                h(err, result);
            };
        wrapper(
            std::move(cb),
            &ymod_imap_client::ImapClient::status,
            name,
            "MESSAGES RECENT UIDNEXT UIDVALIDITY UNSEEN");
    }

    template <typename Handler>
    void select(const ymod_imap_client::Utf8MailboxName& name, Handler h)
    {
        client->setCommandTimeout(get_timeout("select"));
        if (is_selected(name))
        {
            auto handler = std::bind(h, code::ok);
            strand.post(handler);
        }
        else
        {
            imap_call_wrapper<ymod_imap_client::ImapResultPtr> wrapper(client);
            wrapper.logger(context->logger());
            auto cb =
                [this, self = shared_from_this(), h, name](error err, const std::string&) mutable {
                    handle_status(err);
                    if (!err)
                    {
                        mb_name = name;
                        state = state_t::selected;
                    }
                    h(err);
                };
            wrapper(std::move(cb), &ymod_imap_client::ImapClient::select, name);
        }
    }

    template <typename Handler>
    void examine_with_counters(const ymod_imap_client::Utf8MailboxName& name, Handler h)
    {
        client->setCommandTimeout(get_timeout("examine_with_counters"));
        imap_call_wrapper<ymod_imap_client::ImapMailboxResultPtr> wrapper(client);
        wrapper.logger(context->logger());
        auto cb = [this, self = shared_from_this(), h, name](
                      error err,
                      const std::string&,
                      ymod_imap_client::ImapMailboxResultPtr&& result) mutable {
            handle_status(err);
            if (!err)
            {
                mb_name = name;
                state = state_t::examined;
            }
            h(err, result);
        };
        wrapper(std::move(cb), &ymod_imap_client::ImapClient::examine, name);
    }

    template <typename Handler>
    void examine(const ymod_imap_client::Utf8MailboxName& name, Handler h)
    {
        if (is_examined_or_selected(name))
        {
            auto handler = std::bind(h, code::ok);
            strand.post(handler);
        }
        else
        {
            auto cb = [h](error err, ymod_imap_client::ImapMailboxResultPtr) mutable { h(err); };
            examine_with_counters(name, std::move(cb));
        }
    }

    template <typename Handler>
    void uid_fetch(const std::string& seq, Handler h)
    {
        client->setCommandTimeout(get_timeout("uid_fetch"));
        imap_call_wrapper<ymod_imap_client::MessageSetPtr> wrapper(client);
        wrapper.logger(context->logger());
        auto cb =
            [this, self = shared_from_this(), h](
                error err, const std::string&, ymod_imap_client::MessageSetPtr&& result) mutable {
                handle_status(err);
                h(err, result);
            };
        ymod_imap_client::FetchArgs args = {
            ymod_imap_client::FetchArg::Uid,
            ymod_imap_client::FetchArg::Flags,
            ymod_imap_client::FetchArg::Date,
            ymod_imap_client::FetchArg::Size,
        };
        wrapper(std::move(cb), &ymod_imap_client::ImapClient::uidFetch, seq, args);
    }

    template <typename Handler>
    void num_fetch(const std::string& seq, Handler h)
    {
        client->setCommandTimeout(get_timeout("num_fetch"));
        imap_call_wrapper<ymod_imap_client::MessageSetPtr> wrapper(client);
        wrapper.logger(context->logger());
        auto cb =
            [this, self = shared_from_this(), h](
                error err, const std::string&, ymod_imap_client::MessageSetPtr&& result) mutable {
                handle_status(err);
                h(err, result);
            };
        ymod_imap_client::FetchArgs args = {
            ymod_imap_client::FetchArg::Uid,
            ymod_imap_client::FetchArg::Flags,
            ymod_imap_client::FetchArg::Date,
            ymod_imap_client::FetchArg::Size,
        };
        wrapper(std::move(cb), &ymod_imap_client::ImapClient::fetch, seq, args);
    }

    template <typename Handler>
    void uid_fetch_body(const std::string& seq, Handler h)
    {
        client->setCommandTimeout(get_timeout("uid_fetch_body"));
        imap_call_wrapper<ymod_imap_client::MessageSetPtr> wrapper(client);
        wrapper.logger(context->logger());
        auto cb =
            [this, self = shared_from_this(), h](
                error err, const std::string&, ymod_imap_client::MessageSetPtr&& result) mutable {
                handle_status(err);
                h(err, result);
            };
        ymod_imap_client::FetchArgs args = { ymod_imap_client::FetchArg::Body };
        wrapper(std::move(cb), &ymod_imap_client::ImapClient::uidFetch, seq, args);
    }

    template <typename Handler>
    void move_messages(
        const std::string& seq,
        const ymod_imap_client::Utf8MailboxName& mailbox,
        Handler h)
    {
        client->setCommandTimeout(get_timeout("move_messages"));
        imap_call_wrapper<ymod_imap_client::CopyuidResultPtr> wrapper(client);
        wrapper.logger(context->logger());
        auto cb = [this, self = shared_from_this(), h](
                      error err,
                      const std::string&,
                      ymod_imap_client::CopyuidResultPtr&& result) mutable {
            handle_status(err);
            h(err, result);
        };
        wrapper(std::move(cb), &ymod_imap_client::ImapClient::uidMove, seq, mailbox);
    }

    template <typename Handler>
    void copy_messages(
        const std::string& seq,
        const ymod_imap_client::Utf8MailboxName& mailbox,
        Handler h)
    {
        client->setCommandTimeout(get_timeout("copy_messages"));
        imap_call_wrapper<ymod_imap_client::CopyuidResultPtr> wrapper(client);
        wrapper.logger(context->logger());
        auto cb = [this, self = shared_from_this(), h](
                      error err,
                      const std::string&,
                      ymod_imap_client::CopyuidResultPtr&& result) mutable {
            handle_status(err);
            h(err, result);
        };
        wrapper(std::move(cb), &ymod_imap_client::ImapClient::uidCopy, seq, mailbox);
    }

    template <typename Handler>
    void create_mailbox(const ymod_imap_client::Utf8MailboxName& name, Handler h)
    {
        client->setCommandTimeout(get_timeout("create_mailbox"));
        imap_call_wrapper<ymod_imap_client::ImapResultPtr> wrapper(client);
        wrapper.logger(context->logger());
        auto cb = [this, self = shared_from_this(), h](error err, const std::string&) mutable {
            handle_status(err);
            h(err);
        };
        wrapper(std::move(cb), &ymod_imap_client::ImapClient::createMailbox, name);
    }

    template <typename Handler>
    void rename_mailbox(
        const ymod_imap_client::Utf8MailboxName& old_name,
        const ymod_imap_client::Utf8MailboxName& new_name,
        Handler h)
    {
        client->setCommandTimeout(get_timeout("rename_mailbox"));
        imap_call_wrapper<ymod_imap_client::ImapResultPtr> wrapper(client);
        wrapper.logger(context->logger());
        auto cb = [this, self = shared_from_this(), h](error err, const std::string&) mutable {
            handle_status(err);
            h(err);
        };
        wrapper(std::move(cb), &ymod_imap_client::ImapClient::renameMailbox, old_name, new_name);
    }

    template <typename Handler>
    void delete_mailbox(const ymod_imap_client::Utf8MailboxName& name, Handler h)
    {
        client->setCommandTimeout(get_timeout("delete_mailbox"));
        imap_call_wrapper<ymod_imap_client::ImapResultPtr> wrapper(client);
        wrapper.logger(context->logger());
        auto cb =
            [this, self = shared_from_this(), h, name](error err, const std::string&) mutable {
                handle_status(err);
                h(err);
            };
        wrapper(std::move(cb), &ymod_imap_client::ImapClient::deleteMailbox, name);
    }

    template <typename Handler>
    void store(const std::string& seq, const std::string& type, const std::string& flags, Handler h)
    {
        client->setCommandTimeout(get_timeout("store"));
        imap_call_wrapper<ymod_imap_client::ImapResultPtr> wrapper(client);
        wrapper.logger(context->logger());
        auto cb = [this, self = shared_from_this(), h](error err, const std::string&) mutable {
            handle_status(err);
            h(err);
        };
        wrapper(std::move(cb), &ymod_imap_client::ImapClient::uidStore, seq, type, flags);
    }

    template <typename Handler>
    void expunge(Handler h)
    {
        client->setCommandTimeout(get_timeout("expunge"));
        imap_call_wrapper<ymod_imap_client::ImapResultPtr> wrapper(client);
        wrapper.logger(context->logger());
        auto cb = [this, self = shared_from_this(), h](error err, const std::string&) mutable {
            handle_status(err);
            h(err);
        };
        wrapper(std::move(cb), &ymod_imap_client::ImapClient::expunge);
    }

    template <typename Handler>
    void append(
        std::string&& body,
        const ymod_imap_client::Utf8MailboxName& mailbox,
        const std::string& flags,
        const std::string& date,
        Handler h)
    {
        client->setCommandTimeout(get_timeout("append"));
        imap_call_wrapper<ymod_imap_client::AppenduidResultPtr> wrapper(client);
        wrapper.logger(context->logger());
        auto cb = [this, self = shared_from_this(), h](
                      error err,
                      const std::string&,
                      ymod_imap_client::AppenduidResultPtr&& result) mutable {
            handle_status(err);
            h(err, result);
        };
        wrapper(
            std::move(cb),
            &ymod_imap_client::ImapClient::append,
            std::move(body),
            mailbox,
            flags,
            date);
    }

    bool connected()
    {
        return state != state_t::not_connected;
    };
    bool authenticated()
    {
        return state >= state_t::authenticated;
    };
    const ymod_imap_client::Capability& get_capability()
    {
        return capability;
    }

    void handle_status(error err)
    {
        if (err == errc::imap_not_connected)
        {
            state = state_t::not_connected;
        }
    }

    void handle_capability(ymod_imap_client::ImapCapabilityPtr capability)
    {
        this->capability = capability->capability;
    }

    void handle_id(ymod_imap_client::IdResultPtr result)
    {
        auto name_it = result->data.find("name");
        if (name_it == result->data.end())
        {
            return;
        }
        auto provider_it = st->id_responses.find(name_it->second);
        if (provider_it == st->id_responses.end())
        {
            return;
        }
        provider = provider_it->second;
    }

    void reset()
    {
        auto module = yplatform::find<ymod_imap_client::call>("imap_client");
        client = module->clientInstance(context);
        state = state_t::not_connected;
        mb_name = ymod_imap_client::Utf8MailboxName();
        capability = ymod_imap_client::Capability();
    }

    std::string get_provider()
    {
        return provider;
    }

    statistics get_stats()
    {
        ymod_imap_client::Statistics session_stats = client->getStats();
        statistics stats;
        stats.command_count = session_stats.commandCount;
        stats.received_bytes = session_stats.receivedBytes;
        stats.sent_bytes = session_stats.sentBytes;
        return stats;
    }

private:
    bool is_selected(const ymod_imap_client::Utf8MailboxName& name)
    {
        return state == state_t::selected && name.asString() == mb_name.asString();
    }

    bool is_examined_or_selected(const ymod_imap_client::Utf8MailboxName& name)
    {
        return (state == state_t::selected || state == state_t::examined) &&
            name.asString() == mb_name.asString();
    }

    void clarify_error_with_reason(error& code, const std::string& reason)
    {
        if (reason.empty())
        {
            return;
        }

        auto it = st->auth_responses.find(reason);
        if (it == st->auth_responses.end())
        {
            return;
        }

        code = error(it->second, code.message());
    }

    time_traits::duration get_timeout(const std::string& command_name)
    {
        auto it = st->command_timeouts.find(command_name);
        if (it != st->command_timeouts.end())
        {
            return it->second;
        }

        it = st->command_timeouts.find("default");
        return it == st->command_timeouts.end() ? time_traits::duration::max() : it->second;
    };

    string get_provider_by_server(const std::string& server)
    {
        auto it = st->provider_by_server.find(server);
        if (it == st->provider_by_server.end()) return "custom";
        return it->second;
    }

    state_t state = state_t::not_connected;
    boost::asio::io_service::strand strand;
    ymod_imap_client::ImapClientPtr client;
    ymod_imap_client::Utf8MailboxName mb_name;
    ymod_imap_client::Capability capability;
    std::string provider = "unknown";
    context_ptr context;
    settings_ptr st;
};

using imap_wrapper_ptr = std::shared_ptr<imap_wrapper>;

}
