#pragma once

#include "imap_wrapper.h"
#include "external_mailbox_settings.h"

#include <mailbox/data_types/folder.h>
#include <mailbox/common.h>

#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

#include <boost/algorithm/string.hpp>

namespace xeno::mailbox::external {

class mark_flags_op : public yplatform::log::contains_logger
{
    using yield_context = yplatform::yield_context<mark_flags_op>;

public:
    mark_flags_op(
        const path_t& path,
        imap_id_vector_ptr ids,
        const flags_t& add,
        const flags_t& del,
        const without_data_cb& cb,
        imap_wrapper_ptr client,
        settings_ptr settings,
        const yplatform::log::source& logger)
        : yplatform::log::contains_logger(logger)
        , client(client)
        , path(path.to_string(), path.delim)
        , add(add)
        , del(del)
        , cb(cb)
        , settings(settings)
    {
        std::vector<std::string> str_ids;
        str_ids.resize(ids->size());
        std::transform(
            ids->begin(),
            ids->end(),
            str_ids.begin(),
            &boost::lexical_cast<std::string, imap_id_t>);
        seq = boost::algorithm::join(str_ids, ",");
    }

    void operator()(yield_context ctx, error err = {})
    {
        try
        {
            reenter(ctx)
            {
                yield client->select(path, ctx);
                if (err) yield break;

                for (auto& flag : add.system_flags)
                {
                    flags.push_back(system_flag_name(flag));
                }
                if (flags.size())
                {
                    yield client->store(seq, "+FLAGS", boost::algorithm::join(flags, " "), ctx);
                    if (err) yield break;
                    flags.clear();
                }

                for (auto& flag : del.system_flags)
                {
                    flags.push_back(system_flag_name(flag));
                }
                if (flags.size())
                {
                    yield client->store(seq, "-FLAGS", boost::algorithm::join(flags, " "), ctx);
                    if (err) yield break;
                }
            }
        }
        catch (const std::exception& e)
        {
            YLOG_L(error) << "external mailbox mark_flags_op exception: " << e.what();
            err = code::external_mailbox_exception;
        }

        if (ctx.is_complete())
        {
            cb(err);
        }
    }

private:
    imap_wrapper_ptr client;

    ymod_imap_client::Utf8MailboxName path;
    std::string seq;
    flags_t add;
    flags_t del;

    std::vector<std::string> flags;

    without_data_cb cb;
    settings_ptr settings;
};

}
