#pragma once

#include "imap_wrapper.h"
#include "external_mailbox_settings.h"

#include <mailbox/data_types/folder.h>
#include <mailbox/common.h>

#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

#include <boost/algorithm/string.hpp>

namespace xeno::mailbox::external {

class move_messages_op : public yplatform::log::contains_logger
{
    using yield_context = yplatform::yield_context<move_messages_op>;

public:
    move_messages_op(
        const path_t& src,
        const path_t& dst,
        imap_id_vector_ptr ids,
        const imap_ids_transform_cb& cb,
        imap_wrapper_ptr client,
        settings_ptr settings,
        const yplatform::log::source& logger)
        : yplatform::log::contains_logger(logger)
        , client(client)
        , src(src.to_string(), src.delim)
        , dst(dst.to_string(), dst.delim)
        , cb(cb)
        , settings(settings)
    {
        std::vector<std::string> str_ids;
        str_ids.resize(ids->size());
        std::transform(
            ids->begin(),
            ids->end(),
            str_ids.begin(),
            &boost::lexical_cast<std::string, imap_id_t>);
        seq = boost::algorithm::join(str_ids, ",");
    }

    move_messages_op(
        const path_t& src,
        const path_t& dst,
        const std::string& sequence,
        const imap_ids_transform_cb& cb,
        imap_wrapper_ptr client,
        settings_ptr settings,
        const yplatform::log::source& logger)
        : yplatform::log::contains_logger(logger)
        , client(client)
        , src(src.to_string(), src.delim)
        , dst(dst.to_string(), dst.delim)
        , seq(sequence)
        , cb(cb)
        , settings(settings)
    {
    }

    void operator()(yield_context ctx, error err = {})
    {
        try
        {
            reenter(ctx)
            {
                yield client->select(src, ctx);
                if (err) yield break;

                if (client->get_capability().move)
                {
                    yield client->move_messages(seq, dst, ctx);
                    if (err) yield break;
                }
                else
                {
                    yield client->copy_messages(seq, dst, ctx);
                    if (err) yield break;

                    yield client->store(
                        seq, "+FLAGS", system_flag_name(system_flag_t::deleted), ctx);
                    if (err) yield break;

                    yield client->expunge(ctx);
                    if (err) yield break;
                }
            }
        }
        catch (const std::exception& e)
        {
            YLOG_L(error) << "external mailbox move_messages_op exception: " << e.what();
            err = code::external_mailbox_exception;
        }

        if (ctx.is_complete())
        {
            cb(err, result);
        }
    }

    void operator()(yield_context ctx, error err, ymod_imap_client::CopyuidResultPtr copyuid_res)
    {
        if (!err)
        {
            if (copyuid_res->newUids.size() == copyuid_res->oldUids.size())
            {
                for (std::size_t i = 0; i < copyuid_res->newUids.size(); ++i)
                {
                    result->emplace(copyuid_res->oldUids[i], copyuid_res->newUids[i]);
                }
            }
            else
            {
                YLOG_L(error) << "imap parsing error: different sizes in copyuid response old:"
                              << copyuid_res->oldUids.size()
                              << ", new: " << copyuid_res->newUids.size();
            }
        }

        (*this)(ctx, err);
    }

private:
    imap_wrapper_ptr client;

    ymod_imap_client::Utf8MailboxName src;
    ymod_imap_client::Utf8MailboxName dst;
    std::string seq;

    imap_id_transform_map_ptr result = std::make_shared<imap_id_transform_map>();

    imap_ids_transform_cb cb;
    settings_ptr settings;
};

}
