#pragma once

#include <mailbox/common.h>
#include <yplatform/coroutine.h>
#include <yplatform/net/dns/resolver.h>
#include <yplatform/yield.h>

namespace xeno::mailbox::external {

class resolve_server_op : public yplatform::log::contains_logger
{
    using resolver_type = yplatform::net::dns::resolver;
    using yield_context = yplatform::yield_context<resolve_server_op>;
    using resolve_cb = std::function<void(error, const std::string&)>;

public:
    resolve_server_op(
        const std::string& server,
        boost::asio::io_service& io,
        settings_ptr settings,
        const yplatform::log::source& logger,
        const resolve_cb& cb)
        : yplatform::log::contains_logger(logger)
        , server(server)
        , cb(cb)
        , resolver(io)
        , settings(settings)
    {
    }

    void operator()(yield_context ctx, error err = {}, resolver_type::iterator_a resolver_it = {})
    {
        try
        {
            reenter(ctx)
            {
                yield resolver.async_resolve_a(server, ctx);
                if (err)
                {
                    yield resolver.async_resolve_aaaa(server, ctx);
                }
                if (err)
                {
                    YLOG_L(error) << "resolve_server_op: " << err.message();
                    err = code::resolve_error;
                    yield break;
                }

                result = *resolver_it;
                if (is_ip_blacklisted(result))
                {
                    err = code::server_ip_blacklisted;
                }
            }
        }
        catch (const std::exception& e)
        {
            YLOG_L(error) << "external mailbox resolve_server_op exception: " << e.what();
            err = code::external_mailbox_exception;
        }

        if (ctx.is_complete())
        {
            cb(err, result);
        }
    }

    bool is_ip_blacklisted(const std::string& ip)
    {
        auto ip_addr = make_ip_address(ip);
        for (auto&& network : settings->blacklisted_networks)
        {
            if (network.contains(ip_addr)) return true;
        }
        return false;
    }

private:
    std::string server;
    std::string result;
    resolve_cb cb;
    resolver_type resolver;
    settings_ptr settings;
};

}
