#pragma once

#include "imap_wrapper.h"
#include "external_mailbox_settings.h"

#include <auth/get_access_token_op.h>
#include <mailbox/data_types/folder.h>
#include <mailbox/common.h>
#include <ymod_smtpclient/call.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

#include <boost/algorithm/string.hpp>

namespace xeno::mailbox::external {

class send_op
{
    using request = ymod_smtpclient::Request;
    using request_ptr = std::shared_ptr<request>;
    using rcpt_to = ymod_smtpclient::RcptTo;
    using notify_mode = ymod_smtpclient::NotifyMode;
    using smtp_client = ymod_smtpclient::Call;
    using smtp_client_ptr = boost::shared_ptr<smtp_client>;
    using yield_context = yplatform::yield_context<send_op>;

public:
    send_op(
        smtp_client_ptr client,
        yplatform::task_context_ptr context,
        settings_ptr settings,
        const std::string& smtp_login,
        const auth_data& auth,
        const endpoint& smtp_ep,
        const std::string& from,
        const std::vector<std::string>& to,
        std::shared_ptr<std::string> body,
        bool notify,
        const without_data_cb& cb)
        : client(client)
        , task_ctx(context)
        , settings(settings)
        , smtp_login(smtp_login)
        , auth(auth)
        , smtp_ep(smtp_ep)
        , from(from)
        , to(to)
        , body(body)
        , notify(notify)
        , cb(cb)
    {
    }

    void operator()(yield_context yield_ctx, error err = {})
    {
        if (err)
        {
            return cb(err);
        }

        try
        {
            reenter(yield_ctx)
            {
                if (auth.type == auth_type::oauth)
                {
                    yield yplatform::spawn(std::make_shared<auth::get_access_token_op>(
                        task_ctx,
                        auth.oauth_app,
                        auth.smtp_credentials,
                        settings->social_settings,
                        false,
                        yield_ctx));
                }

                smtp_request = make_smtp_request();

                // TODO may be retry?
                yield client->asyncRun(task_ctx, *smtp_request, yield_ctx);
            }
        }
        catch (const std::exception& e)
        {
            YLOG(task_ctx->logger(), error) << "smtp send exception: " << e.what();
            err = code::external_mailbox_exception;
        }

        if (yield_ctx.is_complete())
        {
            cb(err);
        }
    }

    void operator()(
        yield_context yield_ctx,
        ymod_smtpclient::error::Code err,
        ymod_smtpclient::Response response)
    {
        if (err)
        {
            auto reason = get_smtp_error_reason(err, response);
            YLOG(task_ctx->logger(), error) << "smtp send error: " << reason;
            if (response.session && response.session->replyCode / 100 == 4)
            {
                (*this)(yield_ctx, error(code::smtp_send_temp_error, reason));
            }
            else
            {
                (*this)(yield_ctx, error(code::smtp_send_perm_error, reason));
            }
        }
        else
        {
            (*this)(yield_ctx);
        }
    }

    void operator()(yield_context yield_ctx, error err, const std::string& resp, const time_point&)
    {
        if (err)
        {
            YLOG(task_ctx->logger(), error) << "smtp send get access token error";
            (*this)(yield_ctx, code::get_access_token_error);
        }
        else
        {
            access_token = resp;
            (*this)(yield_ctx);
        }
    }

    request_ptr make_smtp_request()
    {
        using endpoint = ymod_smtpclient::SmtpPoint;
        using auth_data = ymod_smtpclient::AuthData;
        using mail_from = ymod_smtpclient::MailFrom;

        request_ptr result = std::make_shared<request>();

        auto& address = result->address;
        address.proto = endpoint::Proto::smtp;
        address.host = smtp_ep.host;
        address.port = smtp_ep.port;

        if (auth.type == auth_type::oauth)
        {
            result->auth.reset(auth_data::XOAUTH2(smtp_login, access_token));
        }
        else
        {
            result->auth.reset(auth_data::BEST(smtp_login, auth.smtp_credentials));
        }

        result->mailfrom = mail_from(from);
        for (auto& email : to)
        {
            rcpt_to recipient(email);
            if (notify)
            {
                recipient.notifyModes = { notify_mode::Failure, notify_mode::Success };
            }
            result->rcpts.push_back(recipient);
        }

        // copy, because client will change body for smtp dot stuffing
        result->message = std::make_shared<std::string>(*body);

        return result;
    }

    std::string get_smtp_error_reason(
        ymod_smtpclient::error::Code err,
        ymod_smtpclient::Response response)
    {
        std::stringstream reason;
        reason << ymod_smtpclient::error::message(err);
        if (response.session)
        {
            reason << ": " << response.session;
        }
        return reason.str();
    }

private:
    smtp_client_ptr client;
    yplatform::task_context_ptr task_ctx;
    settings_ptr settings;
    std::string smtp_login;
    auth_data auth;
    endpoint smtp_ep;
    std::string from;
    std::vector<std::string> to;
    std::shared_ptr<std::string> body;
    bool notify;
    without_data_cb cb;

    std::string access_token;
    request_ptr smtp_request = std::make_shared<request>();
};

}
