#pragma once

#include <common/mail_errors.h>
#include <mailbox/local/get_or_create_labels_op.h>

#include <macs_pg/macs_pg.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

#include <functional>

namespace xeno::mailbox::local {

namespace ph = std::placeholders;

template <typename MacsService>
class change_flags_op : public yplatform::log::contains_logger
{
public:
    using yield_context = yplatform::yield_context<change_flags_op>;
    using labels_t = std::list<macs::Label>;

    change_flags_op(
        const MacsService& service,
        const mid_vector& mids,
        const flags_t& add_flags,
        const flags_t& del_flags,
        const without_data_cb& cb)
        : service_(service), cb_(cb), add_flags_(add_flags), del_flags_(del_flags)
    {
        for (auto& mid : mids)
        {
            mids_.push_back(std::to_string(mid));
        }
    }

    void operator()(yield_context ctx)
    {
        try
        {
            reenter(ctx)
            {
                yield convert_flags_to_labels(ctx.capture(err_, add_labels_), add_flags_);
                if (err_) yield break;

                yield convert_flags_to_labels(ctx.capture(err_, del_labels_), del_flags_);
                if (err_) yield break;

                if (add_labels_.size())
                {
                    yield service_->envelopes().markEnvelopes(mids_, add_labels_, ctx);
                }
                if (del_labels_.size())
                {
                    yield service_->envelopes().unmarkEnvelopes(mids_, del_labels_, ctx);
                }
            }
        }
        catch (const std::exception& e)
        {
            YLOG_L(error) << "change flags exception: " << e.what();
            return cb_(code::local_mailbox_exception);
        }
        if (ctx.is_complete())
        {
            cb_(err_ ? err_ : code::ok);
        }
    }

    void operator()(yield_context ctx, mail_errors::error_code err, macs::Revision)
    {
        if (err)
        {
            YLOG_L(error) << "change flags: can't update envelope labels: "
                          << mail_error_message(err);
            cb_(err.base());
        }
        else
        {
            (*this)(ctx);
        }
    }

private:
    template <typename Callback>
    void convert_flags_to_labels(Callback cb, const flags_t& flags)
    {
        auto coro =
            std::make_shared<get_or_create_labels_op<MacsService>>(service_, flags, std::move(cb));
        coro->logger(logger());
        yplatform::spawn(coro);
    }

    error err_;
    MacsService service_;
    without_data_cb cb_;
    std::list<macs::Mid> mids_;
    flags_t add_flags_;
    flags_t del_flags_;
    labels_t add_labels_;
    labels_t del_labels_;
};

}

#include <yplatform/unyield.h>
