#pragma once

#include <common/http.h>
#include <common/karma.h>
#include <common/json.h>
#include <common/types.h>

#include <ymod_httpclient/cluster_client.h>
#include <yplatform/util/sstream.h>
#include <yplatform/json.h>
#include <yplatform/encoding/url_encode.h>

#include <boost/algorithm/string/join.hpp>

namespace xeno::mailbox::local {

struct check_spam_op
    : std::enable_shared_from_this<check_spam_op>
    , yplatform::log::contains_logger
{
    check_spam_op(
        uid_t uid,
        const karma_t& karma,
        const std::string& from,
        const std::vector<std::string>& to,
        const std::string& client_ip,
        const std::string& request_id,
        std::string body,
        context_ptr context,
        const yplatform::log::source& logger,
        const without_data_cb& cb)
        : yplatform::log::contains_logger(logger)
        , context(context)
        , params(make_params(uid, karma, from, to, client_ip, request_id))
        , body(std::move(body))
        , cb(cb)
    {
    }

    void operator()()
    {
        std::string headers = "Content-Type: text/plain\r\n";
        auto req = http::request_t::POST("/v2/antispam" + params, headers, std::move(body));
        auto client = yplatform::find<yhttp::cluster_client>("check_spam_client");
        client->async_run(
            context,
            std::move(req),
            [this, self = shared_from_this()](error err, yhttp::response res) {
                cb(process_response(err, std::move(res)));
            });
    }

    std::string make_params(
        uid_t uid,
        const karma_t& karma,
        const std::string& from,
        const std::vector<std::string>& to,
        const std::string& client_ip,
        const std::string& request_id)
    {
        auto params = yhttp::url_encode({ { "source", "xeno" },
                                          { "client_ip", client_ip },
                                          { "request_id", request_id },
                                          { "uid", uid },
                                          { "from", from },
                                          { "karma", karma.value },
                                          { "karma_status", karma.status } });

        yplatform::sstream params_stream(params);
        for (auto& rcpt : to)
        {
            params_stream << yhttp::url_encode({ { "to", rcpt } }, '&');
        }
        return params;
    }

    error process_response(error err, yhttp::response res)
    {
        if (err)
        {
            YLOG_L(error) << "check_spam_op request error: " << err.message();
            return code::so_error;
        }

        if (res.status != 200)
        {
            YLOG_L(error) << "check_spam_op bad http response: code=" << res.status
                          << ", body=" << res.body;
            return code::so_error;
        }

        yplatform::json_value response;
        if (auto error = response.parse(res.body, yplatform::json_type::tobject))
        {
            TASK_LOG(context, error) << "check_spam_op json parse error: " << *error;
            return code::so_error;
        }

        auto resolution = response["resolution"].to_string();
        if (resolution == "HAM")
        {
            return {};
        }
        if (resolution == "SPAM")
        {
            return code::message_is_spam;
        }
        TASK_LOG(context, error) << "check_spam_op unknonw so resolution: " << resolution;
        return code::so_error;
    }

    context_ptr context;
    std::string params;
    std::string body;
    without_data_cb cb;
};

}
