#pragma once

#include <common/mail_errors.h>

#include <common/split.h>
#include <macs_pg/macs_pg.h>
#include <yplatform/log.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

namespace xeno::mailbox::local {

template <typename MacsService>
struct clear_folder_op : public yplatform::log::contains_logger
{
    using yield_context = yplatform::yield_context<clear_folder_op>;

    clear_folder_op(
        MacsService service,
        const fid_t& fid,
        size_t delete_messages_chunk,
        const without_data_cb& cb)
        : service(service), fid(fid), delete_messages_chunk(delete_messages_chunk), cb(cb)
    {
    }

    void operator()(yield_context ctx, error ec = {})
    {
        try
        {
            reenter(ctx)
            {
                YLOG_L(info) << "getting folder list";
                service->folders().resetFoldersCache();
                yield service->folders().getAllFolders(ctx);

                if (ec) yield break;

                if (!folders.exists(fid))
                {
                    ec = code::folder_not_found;
                    yield break;
                }

                YLOG_L(info) << "updating downloaded range";
                yield service->mailish().updateDownloadedRange(fid, 0, 0, ctx);
                if (ec) yield break;

                YLOG_L(info) << "getting folder mids";
                yield service->envelopes().getMidsByFolder(fid, ctx);
                if (ec) yield break;

                mid_chunks = split_for_chunks(mids, delete_messages_chunk);
                for (chunks_iterator = mid_chunks.begin(); chunks_iterator != mid_chunks.end();
                     ++chunks_iterator)
                {
                    if (chunks_iterator->size())
                    {
                        YLOG_L(info)
                            << "deleting chunk with " << chunks_iterator->size() << " messages";
                        yield service->envelopes().remove(*chunks_iterator, ctx);
                        if (ec) yield break;
                    }
                }
            }
        }
        catch (const std::exception& e)
        {
            YLOG_L(error) << "clear folder op exception: " << e.what();
            ec = code::local_mailbox_exception;
        }

        if (ctx.is_complete())
        {
            cb(ec);
        }
    }

    void operator()(yield_context ctx, mail_errors::error_code err, macs::FolderSet folders)
    {
        if (!err)
        {
            this->folders = std::move(folders);
        }
        else
        {
            YLOG_L(error) << "clear folder op error: " << mail_error_message(err);
        }
        (*this)(ctx, err.base());
    }

    void operator()(yield_context ctx, mail_errors::error_code err, macs::Revision)
    {
        if (err)
        {
            YLOG_L(error) << "clear folder op error: " << mail_error_message(err);
        }
        (*this)(ctx, err.base());
    }

    void operator()(yield_context ctx, mail_errors::error_code err, macs::MidList mids)
    {
        if (!err)
        {
            this->mids = std::move(mids);
        }
        else
        {
            YLOG_L(error) << "clear folder op error: " << mail_error_message(err);
        }
        (*this)(ctx, err.base());
    }

    MacsService service;
    fid_t fid;
    size_t delete_messages_chunk;
    without_data_cb cb;
    macs::FolderSet folders;
    macs::MidList mids;
    std::vector<macs::MidList> mid_chunks = { macs::MidList() };
    std::vector<macs::MidList>::iterator chunks_iterator;
};

}

#include <yplatform/unyield.h>
