#pragma once

#include "update_folder_op.h"
#include "clear_folder_op.h"
#include "conversions.h"

#include <common/mail_errors.h>

#include <macs_pg/macs_pg.h>

#include <yplatform/log.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

namespace xeno::mailbox::local {

template <typename MacsService>
struct clear_mailbox_op : public yplatform::log::contains_logger
{
    using yield_context = yplatform::yield_context<clear_mailbox_op>;

    clear_mailbox_op(MacsService service, size_t delete_messages_chunk, const without_data_cb& cb)
        : service(service), delete_messages_chunk(delete_messages_chunk), cb(cb)
    {
    }

    void operator()(yield_context ctx, error ec = {})
    {
        try
        {
            reenter(ctx)
            {
                YLOG_L(info) << "getting folder list";
                yield service->folders().getAllFolders(ctx);
                if (ec) yield break;

                for (folder_iterator = folders.begin(); folder_iterator != folders.end();
                     ++folder_iterator)
                {
                    YLOG_L(info) << "clear folder name=" << folder_iterator->second.name()
                                 << ", fid=" << folder_iterator->first;
                    yield clear_folder(ctx, folder_iterator->first);
                    if (ec) yield break;

                    if (!folder_iterator->second.isSystem())
                    {
                        YLOG_L(info) << "deleting folder: name=" << folder_iterator->second.name()
                                     << ", fid=" << folder_iterator->first;
                        yield service->folders().erase(folder_iterator->second, ctx);
                    }
                    else
                    {
                        YLOG_L(info)
                            << "erasing system folder mailish data: name="
                            << folder_iterator->second.name() << ", fid=" << folder_iterator->first;
                        // TODO remove entry from mailish folders
                        yield service->mailish().updateDownloadedRange(
                            folder_iterator->second.fid(), 0, 0, ctx);
                        if (ec) yield break;

                        yield update_folder(ctx, folder_iterator->second);
                    }
                    if (ec) yield break;
                }
            }
        }
        catch (const std::exception& e)
        {
            YLOG_L(error) << "clear mailbox op exception: " << e.what();
            ec = code::local_mailbox_exception;
        }

        if (ctx.is_complete())
        {
            cb(ec);
        }
    }

    void operator()(yield_context ctx, mail_errors::error_code err, macs::FolderSet folders)
    {
        if (!err)
        {
            this->folders = std::move(folders);
        }
        else
        {
            YLOG_L(error) << "clear mailbox op error: " << mail_error_message(err);
        }
        (*this)(ctx, err.base());
    }

    void operator()(yield_context ctx, mail_errors::error_code err, macs::Revision)
    {
        if (err)
        {
            YLOG_L(error) << "clear mailbox op error: " << mail_error_message(err);
        }
        (*this)(ctx, err.base());
    }

    void update_folder(yield_context ctx, const macs::Folder& macs_folder)
    {
        auto folder = to_folder(macs_folder, macs::MailishFolderChunk());
        auto coro = std::make_shared<update_folder_op<MacsService>>(
            service, folder, macs_folder.parentId(), ctx);
        coro->logger(logger());
        yplatform::spawn(coro);
    }

    void clear_folder(yield_context ctx, const fid_t& fid)
    {
        auto coro = std::make_shared<clear_folder_op<MacsService>>(
            service, fid, delete_messages_chunk, ctx);
        coro->logger(logger());
        yplatform::spawn(coro);
    }

    MacsService service;
    size_t delete_messages_chunk;
    without_data_cb cb;

    macs::FolderSet folders;
    macs::FolderSet::iterator folder_iterator;
};

}

#include <yplatform/unyield.h>
