#pragma once

#include <common/mail_errors.h>

#include <macs_pg/macs_pg.h>
#include <yplatform/log.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

namespace xeno::mailbox::local {

template <typename MacsService>
struct delete_folder_op : public yplatform::log::contains_logger
{
    using yield_context = yplatform::yield_context<delete_folder_op>;

    delete_folder_op(MacsService service, const fid_t& fid, const without_data_cb& cb)
        : service(service), fid(fid), cb(cb)
    {
    }

    void operator()(yield_context ctx, error ec = {})
    {
        try
        {
            reenter(ctx)
            {
                YLOG_L(info) << "getting folder list";
                service->folders().resetFoldersCache();
                yield service->folders().getAllFolders(ctx);

                if (ec)
                {
                    yield break;
                }

                if (!folders.exists(fid))
                {
                    ec = code::folder_not_found;
                    yield break;
                }

                YLOG_L(info) << "deleting folder";
                yield service->folders().erase(folders.at(fid), ctx);
            }
        }
        catch (const std::exception& e)
        {
            YLOG_L(error) << "delete folder op exception: " << e.what();
            ec = code::local_mailbox_exception;
        }

        if (ctx.is_complete())
        {
            cb(ec);
        }
    }

    void operator()(yield_context ctx, mail_errors::error_code err, macs::FolderSet folders)
    {
        if (!err)
        {
            this->folders = std::move(folders);
        }
        else
        {
            YLOG_L(error) << "delete folder op error: " << mail_error_message(err);
        }
        (*this)(ctx, err.base());
    }

    void operator()(yield_context ctx, mail_errors::error_code err, macs::Revision)
    {
        if (err)
        {
            YLOG_L(error) << "delete folder op error: " << mail_error_message(err);
        }
        (*this)(ctx, err.base());
    }

    MacsService service;
    fid_t fid;
    without_data_cb cb;
    macs::FolderSet folders;
};

}

#include <yplatform/unyield.h>
