#pragma once

#include <mailbox/common.h>
#include <common/http.h>
#include <common/json.h>
#include <common/types.h>

#include <ymod_httpclient/cluster_client.h>
#include <ymod_httpclient/url_encode.h>
#include <yplatform/log.h>

#include <sstream>
#include <vector>

namespace xeno::mailbox::local {

struct get_attachments_sids_op
    : std::enable_shared_from_this<get_attachments_sids_op>
    , yplatform::log::contains_logger
{
    get_attachments_sids_op(
        context_ptr context,
        uid_t uid,
        const std::string& user_ticket,
        mid_t mid,
        const std::vector<std::string>& hids,
        const sids_cb& cb)
        : context(context)
        , user_ticket(user_ticket)
        , uid(uid)
        , mid(mid)
        , hids(hids)
        , url("/v2/attach_sid")
        , cb(cb)
    {
    }

    void operator()()
    {
        auto request = http::request_t::GET(url + make_params(uid, mid, hids));
        http::append_tvm2_user_header(request, user_ticket);
        auto client = yplatform::find<yhttp::cluster_client>("hound_client");
        client->async_run(
            context,
            std::move(request),
            [this, self = shared_from_this()](error err, yhttp::response res) {
                error ec = code::ok;
                std::vector<std::string> sids;
                try
                {
                    if (err || res.status / 100 != 2)
                    {
                        auto error_msg = (err ? err.message() : parse_error_response(res.body));
                        YLOG_L(error)
                            << "get attachments sids error: "
                            << (error_msg.size() ? error_msg :
                                                   std::to_string(res.status) + " " + res.reason);
                        ec = code::cannot_get_attachments_sids;
                    }
                    else
                    {
                        sids = parse_response(res.body);
                    }
                }
                catch (const std::exception& e)
                {
                    YLOG_L(error) << "get attachments sids exception: " << e.what();
                    ec = code::cannot_get_attachments_sids;
                }
                cb(ec, std::move(sids));
            });
    }

    std::vector<std::string> parse_response(const std::string& response)
    {
        json::value sids;
        json::reader reader;
        std::vector<std::string> result;
        result.reserve(hids.size());
        if (reader.parse(response, sids))
        {
            for (auto& hid : hids)
            {
                if (sids.isMember(hid))
                {
                    result.emplace_back(sids[hid].asString());
                }
                else
                {
                    throw std::runtime_error("no sid for hid=" + hid);
                }
            }
        }
        return result;
    }

    std::string parse_error_response(const std::string& response)
    {
        json::value result;
        json::reader reader;
        std::stringstream error_msg;
        if (reader.parse(response, result))
        {
            auto& error = result["error"];
            error_msg << "code=" << error["code"].asString()
                      << " message=" << error["message"].asString()
                      << " reason=" << error["reason"].asString();
        }
        return error_msg.str();
    }

    std::string make_params(uid_t uid, mid_t mid, const std::vector<std::string>& hids)
    {
        std::stringstream result;
        result << yhttp::url_encode({ { "uid", uid }, { "mid", mid } });
        for (auto& hid : hids)
        {
            result << yhttp::url_encode({ { "hids", hid } }, '&');
        }
        return result.str();
    }

    context_ptr context;
    std::string user_ticket;
    uid_t uid;
    mid_t mid;
    std::vector<std::string> hids;
    std::string url;
    sids_cb cb;
};

}
