#pragma once

#include <common/mail_errors.h>

#include <macs_pg/macs_pg.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

namespace xeno::mailbox::local {

struct flags_by_lids
{
    std::map<lid, system_flag_t> system_flags;
    std::map<lid, std::string> user_flags;
};

template <typename MacsService>
struct get_flags_by_lids_op : public yplatform::log::contains_logger
{
    using yield_context = yplatform::yield_context<get_flags_by_lids_op>;
    using flags_by_lids_cb = std::function<void(error, flags_by_lids)>;

    get_flags_by_lids_op(
        const MacsService& service,
        const lid_vector& lids,
        const flags_by_lids_cb& cb)
        : service(service), lids(lids), cb(cb)
    {
    }

    void operator()(yield_context ctx, error ec = code::ok)
    {
        if (ec) return cb(ec, {});

        try
        {
            reenter(ctx)
            {
                yield service->labels().getAllLabels(ctx);

                macs::LabelSet requested_labels;
                for (auto& lid : lids)
                {
                    auto label = all_labels.find(lid);
                    if (label != all_labels.end())
                    {
                        requested_labels.insert(*label);
                    }
                    else
                    {
                        ec = code::label_not_found;
                        break;
                    }
                }

                if (!ec)
                {
                    result = macs_labels_to_flags(requested_labels);
                }
            }
        }
        catch (const std::exception& e)
        {
            YLOG_L(error) << "get flags by lids exception: " << e.what();
            ec = code::local_mailbox_exception;
        }
        if (ctx.is_complete())
        {
            cb(ec, result);
        }
    }

    void operator()(yield_context ctx, mail_errors::error_code err, macs::LabelSet labels)
    {
        if (err)
        {
            YLOG_L(error) << "can't get all labels: " << mail_error_message(err);
            return (*this)(ctx, err.base());
        }

        all_labels = std::move(labels);
        (*this)(ctx);
    }

    flags_by_lids macs_labels_to_flags(const macs::LabelSet& labels)
    {
        flags_by_lids res;
        for (auto& [lid, macs_label] : labels)
        {
            if (macs_label.isSystem())
            {
                auto flag_opt = macs_symbol_to_system_flag(macs_label.symbolicName());
                if (flag_opt)
                {
                    res.system_flags.insert({ lid, *flag_opt });
                }
            }
            else if (macs_label.isUser())
            {
                res.user_flags.insert({ lid, macs_label.name() });
            }
        }

        return res;
    }

    boost::optional<system_flag_t> macs_symbol_to_system_flag(const macs::Label::Symbol& symbol)
    {
        if (symbol == macs::Label::Symbol::seen_label)
        {
            return system_flag_t::seen;
        }
        if (symbol == macs::Label::Symbol::answered_label)
        {
            return system_flag_t::answered;
        }
        if (symbol == macs::Label::Symbol::important_label)
        {
            return system_flag_t::flagged;
        }
        if (symbol == macs::Label::Symbol::deleted_label)
        {
            return system_flag_t::deleted;
        }
        if (symbol == macs::Label::Symbol::draft_label)
        {
            return system_flag_t::draft;
        }

        return boost::optional<system_flag_t>();
    }

    MacsService service;
    lid_vector lids;
    flags_by_lids_cb cb;
    macs::LabelSet all_labels;
    flags_by_lids result;
};

}

#include <yplatform/unyield.h>
