#pragma once

#include <common/mail_errors.h>

#include <macs_pg/macs_pg.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

namespace xeno::mailbox::local {

template <typename MacsService>
struct get_messages_flags_op : public yplatform::log::contains_logger
{
    using yield_context = yplatform::yield_context<get_messages_flags_op>;

    get_messages_flags_op(
        const MacsService& service,
        message_vector_ptr messages,
        const messages_vector_cb& cb)
        : service(service), messages(messages), cb(cb)
    {
        // some messages don't contains mid
        // we should filter them out before request
        for (auto& message : *messages)
        {
            if (message.mid)
            {
                mids.push_back(std::to_string(message.mid));
            }
        }
    }

    void operator()(yield_context ctx, error ec = code::ok)
    {
        if (ec) return cb(ec, nullptr);

        try
        {
            reenter(ctx)
            {
                if (mids.empty()) yield break;

                yield service->envelopes().getByIds(mids, ctx);
                for (labels_it = message_labels.begin(); labels_it != message_labels.end();
                     ++labels_it)
                {
                    all_lids.insert(labels_it->second.begin(), labels_it->second.end());
                }
                yield get_flags_by_lids(ctx);
            }
        }
        catch (const std::exception& e)
        {
            YLOG_L(error) << "get message info exception: " << e.what();
            ec = code::local_mailbox_exception;
        }
        if (ctx.is_complete())
        {
            cb(ec, messages);
        }
    }

    // handler called per item and then last call with empty item
    void operator()(
        yield_context ctx,
        mail_errors::error_code err,
        io_result::Cursor<macs::Envelope> envelope)
    {
        if (err)
        {
            YLOG_L(error) << "can't get envelopes: " << mail_error_message(err);
            return (*this)(ctx, err.base());
        }

        if (!envelope)
        {
            if (message_labels.size() != mids.size())
            {
                return (*this)(ctx, code::message_not_found);
            }
            return (*this)(ctx);
        }

        message_labels[envelope->mid()] = envelope->labels();
    }

    void operator()(yield_context ctx, error err, const flags_by_lids& flags_by_lids)
    {
        if (err)
        {
            return (*this)(ctx, err);
        }

        for (labels_it = message_labels.begin(); labels_it != message_labels.end(); ++labels_it)
        {
            mid_t current_mid = std::stoull(labels_it->first);
            auto it =
                std::find_if(messages->begin(), messages->end(), [current_mid](const message& msg) {
                    return msg.mid == current_mid;
                });

            if (it == messages->end())
            {
                return (*this)(ctx, code::message_not_found);
            }

            it->flags = flags_t();
            for (const auto& lid : labels_it->second)
            {
                auto iter = flags_by_lids.system_flags.find(lid);
                if (iter != flags_by_lids.system_flags.end())
                {
                    it->flags.system_flags.insert(iter->second);
                }
            }

            for (const auto& lid : labels_it->second)
            {
                auto iter = flags_by_lids.user_flags.find(lid);
                if (iter != flags_by_lids.user_flags.end())
                {
                    it->flags.user_flags.insert(iter->second);
                }
            }
        }
        (*this)(ctx);
    }

    void get_flags_by_lids(yield_context ctx)
    {
        lid_vector lids;
        lids.reserve(all_lids.size());
        std::copy(all_lids.begin(), all_lids.end(), std::back_inserter(lids));
        auto op = std::make_shared<get_flags_by_lids_op<MacsService>>(service, lids, ctx);
        op->logger(logger());
        yplatform::spawn(op);
    }

    MacsService service;
    message_vector_ptr messages;
    messages_vector_cb cb;

    macs::MidList mids;
    using mid_labels_map = std::map<macs::Mid, lid_vector>;
    mid_labels_map message_labels;
    mid_labels_map::iterator labels_it;
    std::set<lid> all_lids;
};

}

#include <yplatform/unyield.h>
