#pragma once

#include <common/mail_errors.h>

#include <macs_pg/macs_pg.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

namespace xeno::mailbox::local {

template <typename MacsService>
class get_or_create_label_op : public yplatform::log::contains_logger
{
public:
    using yield_context = yplatform::yield_context<get_or_create_label_op>;
    using label_t = macs::Label;
    using label_opt = std::optional<label_t>;
    using callback_t = std::function<void(error, label_t)>;

    get_or_create_label_op(
        const MacsService& service,
        const std::string& name,
        const std::string& color,
        const std::string& type,
        const callback_t& cb,
        const yplatform::log::source& logger = {})
        : yplatform::log::contains_logger(logger)
        , service_(service)
        , cb_(cb)
        , name_(name)
        , color_(color)
        , type_(type)
    {
    }

    get_or_create_label_op(
        const MacsService& service,
        const std::string& symbol,
        const callback_t& cb,
        const yplatform::log::source& logger = {})
        : yplatform::log::contains_logger(logger), service_(service), cb_(cb), symbol_(symbol)
    {
    }

    void operator()(yield_context ctx, error ec = {})
    {
        try
        {
            reenter(ctx)
            {
                yield service_->labels().getAllLabels(ctx);
                if (ec) yield break;

                if (symbol_.empty())
                {
                    if (result_ = get_if_exist(name_, label_t::Type::getByTitle(type_)))
                        yield break;

                    yield service_->labels().createLabelWithType(name_, color_, type_, ctx);
                }
                else
                {
                    if (result_ = get_if_exist(label_t::Symbol::getByTitle(symbol_))) yield break;

                    yield service_->labels().createLabel(label_t::Symbol::getByTitle(symbol_), ctx);
                }
            }
        }
        catch (const std::exception& e)
        {
            YLOG_L(error) << "get or create label exception: " << e.what();
            return cb_(code::local_mailbox_exception, label_t());
        }
        if (ctx.is_complete())
        {
            cb_(ec, result_ ? std::move(*result_) : label_t());
        }
    }

    void operator()(yield_context ctx, mail_errors::error_code err, macs::LabelSet labels)
    {
        if (err)
        {
            YLOG_L(error) << "can't get all labels: " << mail_error_message(err);
            return (*this)(ctx, err.base());
        }

        all_labels_ = std::move(labels);
        (*this)(ctx);
    }

    void operator()(yield_context ctx, mail_errors::error_code err, label_t label)
    {
        if (err)
        {
            YLOG_L(error) << "can't get or create label: " << mail_error_message(err);
            return (*this)(ctx, err.base());
        }

        result_ = std::move(label);
        (*this)(ctx);
    }

private:
    template <typename... Args>
    label_opt get_if_exist(Args&&... args)
    {
        auto it = all_labels_.find(std::forward<Args>(args)...);
        if (it != all_labels_.end())
        {
            return it->second;
        }
        return label_opt{};
    }

    MacsService service_;
    callback_t cb_;
    std::string name_;
    std::string color_;
    std::string type_;
    std::string symbol_;
    label_opt result_;
    macs::LabelSet all_labels_;
};

}

#include <yplatform/unyield.h>
