#pragma once

#include "get_or_create_label_op.h"

#include <macs_pg/macs_pg.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

namespace xeno::mailbox::local {

namespace ph = std::placeholders;

template <typename MacsService>
class get_or_create_labels_op : public yplatform::log::contains_logger
{
public:
    using yield_context = yplatform::yield_context<get_or_create_labels_op>;
    using labels_t = std::list<macs::Label>;
    using callback_t = std::function<void(error, labels_t)>;

    get_or_create_labels_op(const MacsService& service, const flags_t& flags, const callback_t& cb)
        : service_(service)
        , cb_(cb)
        , user_flags_(flags.user_flags)
        , system_flags_(flags.system_flags)
    {
    }

    void operator()(yield_context ctx, error ec = {})
    {
        try
        {
            reenter(ctx)
            {
                while (system_flags_.size())
                {
                    yield yplatform::spawn(std::make_shared<get_or_create_label_op<MacsService>>(
                        service_,
                        symbol_from_system_flag(*system_flags_.begin()).title(),
                        ctx,
                        logger()));
                    if (ec) yield break;

                    system_flags_.erase(system_flags_.begin());
                }

                while (user_flags_.size())
                {
                    yield yplatform::spawn(std::make_shared<get_or_create_label_op<MacsService>>(
                        service_, *user_flags_.begin(), "", "user", ctx, logger()));
                    if (ec) yield break;

                    user_flags_.erase(user_flags_.begin());
                }
            }
        }
        catch (const std::exception& e)
        {
            YLOG_L(error) << "get or create labels exception: " << e.what();
            return cb_(code::local_mailbox_exception, labels_t());
        }
        if (ctx.is_complete())
        {
            cb_(ec, std::move(result_));
        }
    }

    void operator()(yield_context ctx, error ec, macs::Label label)
    {
        if (!ec)
        {
            result_.push_back(std::move(label));
        }
        (*this)(ctx, ec);
    }

private:
    using system_flags_t = flags_t::system_flags_t;
    using user_flags_t = flags_t::user_flags_t;

    macs::Label::Symbol symbol_from_system_flag(const system_flag_t& flag)
    {
        using sym = macs::Label::Symbol;
        switch (flag)
        {
        case system_flag_t::seen:
            return sym::seen_label;
        case system_flag_t::answered:
            return sym::answered_label;
        case system_flag_t::deleted:
            return sym::deleted_label;
        case system_flag_t::draft:
            return sym::draft_label;
        case system_flag_t::recent:
            return sym::recent_label;
        case system_flag_t::flagged:
            return sym::important_label;
        default:
            throw std::runtime_error("symbol_from_system_flag: unknown flag");
        }
    }

    MacsService service_;
    callback_t cb_;
    user_flags_t user_flags_;
    system_flags_t system_flags_;
    labels_t result_;
};

}

#include <yplatform/unyield.h>
