#pragma once

#include "../common.h"
#include "store_message_response.h"
#include "update_flags_op.h"
#include "change_flags_op.h"
#include "update_folder_op.h"
#include "check_spam_op.h"
#include "get_flags_by_lids_op.h"
#include "get_messages_flags_op.h"
#include "sendbernar_client.h"
#include "store_message_op.h"
#include "clear_mailbox_op.h"
#include "get_attachments_sids_op.h"
#include "delete_folder_op.h"
#include "clear_folder_op.h"
#include "get_or_create_label_op.h"
#include "update_label_op.h"
#include "set_folder_symbol_op.h"
#include "settings.h"
#include "conversions.h"

#include <auth/get_user_info.h>
#include <common/http.h>
#include <common/mail_errors.h>
#include <common/server_info.h>
#include <mdb/accounts_repository.h>

#include <macs_pg/macs_pg.h>
#include <macs/folders_repository.h>
#include <macs/data/folder_sort_options.h>
#include <macs_pg/mailish/entry_factory.h>
#include <macs_pg/mailish/move_coords_factory.h>
#include <macs_pg/mailish/repository.h>
#include <mail/ymod_cachedb/include/cache.h>

#include <yplatform/coroutine.h>
#include <yplatform/module.h>

#include <boost/algorithm/string.hpp>

namespace xeno::mailbox::local {

namespace ph = std::placeholders;

template <typename MacsService>
class local_mailbox_impl
    : public std::enable_shared_from_this<local_mailbox_impl<MacsService>>
    , public yplatform::log::contains_logger
{
public:
    using context_ptr = yplatform::task_context_ptr;
    using accounts_repository_ptr = mdb::accounts_repository_ptr;

    local_mailbox_impl(
        boost::asio::io_service& io,
        const yplatform::log::source& logger,
        context_ptr context,
        MacsService service,
        accounts_repository_ptr accounts_repository,
        const uid_t uid,
        local::settings_ptr settings)
        : yplatform::log::contains_logger(logger)
        , io(io)
        , context(context)
        , meta_backend(service)
        , accounts_repository(accounts_repository)
        , sendbernar(std::make_shared<sendbernar_client>(context, logger))
        , uid(uid)
        , settings(settings)
    {
        if (!service)
        {
            throw std::runtime_error("local_mailbox_impl: service can't be empty");
        }
    }

    void get_folder_vector(const folder_vector_cb& cb)
    {
        meta_backend->folders().getAllFolders(
            [this, self = yplatform::shared_from(this), cb](
                mail_errors::error_code err, macs::FolderSet folder_set) {
                if (err)
                {
                    YLOG_L(error) << "can't get all folders: " << mail_error_message(err);
                    return cb(err.base(), std::make_shared<folder_vector>());
                }
                meta_backend->mailish().getFolders(
                    [this, self, folder_set = std::move(folder_set), cb](
                        mail_errors::error_code err, macs::MailishFolderChunk mailish_folders) {
                        auto result = std::make_shared<folder_vector>();
                        if (err)
                        {
                            YLOG_L(error) << "can't get folders: " << mail_error_message(err);
                            return cb(err.base(), result);
                        }
                        for (const auto& [fid, folder] : folder_set)
                        {
                            result->push_back(to_folder(folder, mailish_folders));
                        }
                        cb(code::ok, result);
                    });
            });
    }

    void get_messages_info_by_id(
        const fid_t& fid,
        const imap_id_vector& imap_ids,
        msg_info_type info_type,
        const messages_vector_cb& cb)
    {
        std::vector<uint64_t> ids;
        ids.reserve(imap_ids.size());
        std::transform(imap_ids.begin(), imap_ids.end(), std::back_inserter(ids), [](imap_id_t id) {
            return id;
        });
        meta_backend->mailish().getMessages(
            fid,
            ids,
            [this, self = yplatform::shared_from(this), cb, info_type](
                mail_errors::error_code err, const macs::MailishMessageChunk& mailish_messages) {
                try
                {
                    if (err)
                    {
                        YLOG_L(error) << "can't get messages: " << mail_error_message(err);
                        return yplatform::safe_call(context, cb, err.base(), nullptr);
                    }

                    auto messages = std::make_shared<message_vector>();
                    messages->reserve(mailish_messages.size());
                    std::transform(
                        mailish_messages.begin(),
                        mailish_messages.end(),
                        std::back_inserter(*messages),
                        to_message);

                    if (info_type == mailbox::msg_info_type::without_flags || messages->empty())
                    {
                        yplatform::safe_call(context, cb, code::ok, std::move(messages));
                    }
                    else
                    {
                        spawn<get_messages_flags_op<MacsService>>(meta_backend, messages, cb);
                    }
                }
                catch (const std::exception& e)
                {
                    YLOG_L(error) << "get_messages_info_by_id exception: " << e.what();
                    yplatform::safe_call(context, cb, code::local_mailbox_exception, nullptr);
                }
            });
    }

    void get_messages_info_chunk_by_id(
        const fid_t& fid,
        imap_id_t highest_imap_id,
        size_t count,
        const messages_vector_cb& cb)
    {
        meta_backend->mailish().getMessages(
            fid,
            highest_imap_id,
            static_cast<uint32_t>(count),
            [this, self = yplatform::shared_from(this), cb](
                mail_errors::error_code err, const macs::MailishMessageChunk& mailish_messages) {
                try
                {
                    if (err)
                    {
                        YLOG_L(error) << "can't get messages: " << mail_error_message(err);
                        return yplatform::safe_call(context, cb, err.base(), nullptr);
                    }

                    auto messages = std::make_shared<message_vector>();
                    messages->reserve(mailish_messages.size());
                    std::transform(
                        mailish_messages.begin(),
                        mailish_messages.end(),
                        std::back_inserter(*messages),
                        to_message);

                    if (messages->empty())
                    {
                        return yplatform::safe_call(context, cb, code::ok, messages);
                    }

                    spawn<get_messages_flags_op<MacsService>>(meta_backend, messages, cb);
                }
                catch (const std::exception& e)
                {
                    YLOG_L(error) << "get_messages_info_chunk_by_id exception: " << e.what();
                    yplatform::safe_call(context, cb, code::local_mailbox_exception, nullptr);
                }
            });
    }

    void get_messages_info_top(const fid_t& fid, size_t num, const messages_vector_cb& cb)
    {
        meta_backend->mailish().getMessagesTop(
            fid,
            static_cast<uint32_t>(num),
            [this, self = yplatform::shared_from(this), cb](
                mail_errors::error_code err, const macs::MailishMessageChunk& mailish_messages) {
                try
                {
                    if (err)
                    {
                        YLOG_L(error) << "can't get messages: " << mail_error_message(err);
                        return yplatform::safe_call(context, cb, err.base(), nullptr);
                    }

                    auto messages = std::make_shared<message_vector>();
                    messages->reserve(mailish_messages.size());
                    std::transform(
                        mailish_messages.begin(),
                        mailish_messages.end(),
                        std::back_inserter(*messages),
                        to_message);

                    if (messages->empty())
                    {
                        return yplatform::safe_call(context, cb, code::ok, messages);
                    }

                    spawn<get_messages_flags_op<MacsService>>(meta_backend, messages, cb);
                }
                catch (const std::exception& e)
                {
                    YLOG_L(error) << "get_messages_info_top exception: " << e.what();
                    yplatform::safe_call(context, cb, code::local_mailbox_exception, nullptr);
                }
            });
    }

    void get_messages_info_without_flags_by_mid(
        const mid_vector& mids,
        const messages_vector_cb& cb)
    {
        std::vector<std::string> macs_mids;
        macs_mids.reserve(mids.size());
        std::transform(mids.begin(), mids.end(), std::back_inserter(macs_mids), [](mid_t mid) {
            return std::to_string(mid);
        });
        auto requested_count = macs_mids.size();
        meta_backend->mailish().getMessages(
            macs_mids,
            [this, self = yplatform::shared_from(this), cb, requested_count](
                mail_errors::error_code err, const macs::MailishMessageChunk& mailish_messages) {
                try
                {
                    if (err)
                    {
                        YLOG_L(error) << "can't get messages: " << mail_error_message(err);
                        return yplatform::safe_call(context, cb, err.base(), nullptr);
                    }

                    if (mailish_messages.size() != requested_count)
                    {
                        return yplatform::safe_call(context, cb, code::message_not_found, nullptr);
                    }

                    auto messages = std::make_shared<message_vector>();
                    messages->reserve(mailish_messages.size());
                    std::transform(
                        mailish_messages.begin(),
                        mailish_messages.end(),
                        std::back_inserter(*messages),
                        to_message);

                    yplatform::safe_call(context, cb, code::ok, std::move(messages));
                }
                catch (const std::exception& e)
                {
                    YLOG_L(error) << "get_messages_info_without_flags_by_mids exception: "
                                  << e.what();
                    yplatform::safe_call(context, cb, code::local_mailbox_exception, nullptr);
                }
            });
    }

    void get_not_downloaded_messages(uint32_t num, const messages_vector_cb& cb)
    {
        meta_backend->mailish().getNotDownloadedMessages(
            num,
            [this, self = yplatform::shared_from(this), cb](
                mail_errors::error_code err, const macs::MailishMessageChunk& mailish_messages) {
                try
                {
                    if (err)
                    {
                        YLOG_L(error) << "can't get messages: " << mail_error_message(err);
                        return yplatform::safe_call(context, cb, err.base(), nullptr);
                    }

                    auto messages = std::make_shared<message_vector>();
                    messages->reserve(mailish_messages.size());
                    std::transform(
                        mailish_messages.begin(),
                        mailish_messages.end(),
                        std::back_inserter(*messages),
                        to_message);

                    yplatform::safe_call(context, cb, code::ok, std::move(messages));
                }
                catch (const std::exception& e)
                {
                    YLOG_L(error) << "get_not_downloaded_messages exception: " << e.what();
                    yplatform::safe_call(context, cb, code::local_mailbox_exception, nullptr);
                }
            });
    }

    void get_mids_by_tids(const tid_vector& tids, const mid_vector_cb& cb)
    {
        if (tids.empty())
        {
            return cb(code::ok, std::make_shared<mid_vector>());
        }

        macs::TidList macs_tids;
        for (auto tid : tids)
        {
            macs_tids.push_back(std::to_string(tid));
        }

        meta_backend->threads().fillIdsList(
            macs::MidList(),
            macs_tids,
            [cb, this, self = yplatform::shared_from(this)](
                mail_errors::error_code err, const macs::MidList& mids) {
                auto mids_res = std::make_shared<mid_vector>();
                if (err)
                {
                    YLOG_L(error) << "can't get mids by tids: " << mail_error_message(err);
                    return cb(err.base(), mids_res);
                }

                for (auto& macs_mid : mids)
                {
                    mids_res->push_back(std::stoull(macs_mid));
                }

                if (mids_res->size())
                {
                    cb(code::ok, mids_res);
                }
                else
                {
                    cb(code::threads_not_found, mids_res);
                }
            });
    }

    void get_flags_by_lids(const lid_vector& lids, const flags_cb& cb)
    {
        spawn<get_flags_by_lids_op<MacsService>>(
            meta_backend,
            lids,
            [this, self = yplatform::shared_from(this), cb](
                error err, const flags_by_lids& flags_by_lids) {
                if (err)
                {
                    YLOG_L(error) << "can't get flags by lids: " << err.message();
                    return cb(err, nullptr);
                }
                std::set<system_flag_t> system_flags;
                for (const auto& [lid, system_flag] : flags_by_lids.system_flags)
                {
                    system_flags.insert(system_flag);
                }

                std::set<std::string> user_flags;
                for (const auto& [lid, user_flag] : flags_by_lids.user_flags)
                {
                    user_flags.insert(user_flag);
                }
                cb(err, std::make_shared<flags_t>(system_flags, user_flags));
            });
    }

    void delete_messages_by_id(
        const fid_t& fid,
        const imap_id_vector& ids,
        const without_data_cb& cb)
    {
        get_messages_info_by_id(
            fid,
            ids,
            msg_info_type::without_flags,
            [this, self = yplatform::shared_from(this), cb](
                error err, message_vector_ptr messages) {
                if (err)
                {
                    YLOG_L(error) << "can't delete messages: cannot get mids by ids: "
                                  << err.message();
                    return cb(err);
                }
                mid_vector mids;
                mids.reserve(messages->size());
                for (auto& message : *messages)
                {
                    mids.push_back(message.mid);
                }
                delete_messages_by_mid(mids, cb);
            });
    }

    void delete_messages_by_mid(const mid_vector& mids_, const without_data_cb& cb)
    { // TODO unused fid
        macs::MidList mids;
        for (auto& mid : mids_)
        {
            mids.push_back(std::to_string(mid));
        }
        meta_backend->envelopes().remove(
            mids,
            [this, self = yplatform::shared_from(this), cb](
                mail_errors::error_code err, macs::Revision) {
                if (err)
                {
                    YLOG_L(error) << "can't delete messages: " << mail_error_message(err);
                    return cb(err.base());
                }
                cb(code::ok);
            });
    }

    void move_messages(
        const fid_t& src_fid,
        const fid_t& dst_fid,
        tab_opt dst_tab,
        const move_coordinates_vec& move_coords,
        const without_data_cb& cb)
    {
        macs::MailishMoveCoordsChunk chunk;
        chunk.reserve(move_coords.size());
        for (auto& coords : move_coords)
        {
            macs::MailishMoveCoordsFactory factory;
            factory.curImapId(coords.cur_imap_id)
                .newImapId(coords.new_imap_id)
                .mid(std::to_string(coords.mid));
            chunk.push_back(factory.release());
        }

        meta_backend->mailish().moveMessages(
            src_fid,
            dst_fid,
            to_macs_tab(dst_tab),
            chunk,
            [this, self = yplatform::shared_from(this), cb](mail_errors::error_code err) {
                if (err)
                {
                    YLOG_L(error) << "can't move messages: " << mail_error_message(err);
                    cb(err.base());
                }
                else
                {
                    cb(code::ok);
                }
            });
    }

    void init_system_folder(const folder& folder, const without_data_cb& cb)
    {
        macs::MailishFolderInfoFactory factory;
        factory.externalPath(folder.path.to_string()).uidValidity(folder.uidvalidity);
        meta_backend->mailish().initFolder(
            folder.fid,
            factory.release(),
            [this, self = yplatform::shared_from(this), cb](mail_errors::error_code err) {
                if (err)
                {
                    YLOG_L(error) << "can't init system folder: " << mail_error_message(err);
                    cb(err.base());
                }
                else
                {
                    cb(code::ok);
                }
            });
    }

    void create_folder(
        const folder& folder,
        const fid_t& parent,
        const std::string& symbol,
        const folder_vector_cb& cb)
    {
        if (has_parent(folder.path) && parent.empty())
        {
            return cb(code::need_parent_id, folder_vector_ptr());
        }
        macs::MailishFolderInfoFactory factory;
        factory.externalPath(folder.path.to_string()).uidValidity(folder.uidvalidity);
        auto name = folder.path.get_name();
        static const std::set<std::string> system_folders = {
            "Sent", "Drafts", "Trash", "Spam", "Outbox"
        };
        if (system_folders.count(name))
        {
            // while system folders have localization there are may be user folders "Sent",
            // "Drafts", etc convert name to lower case to resolve conflicts with our system folders
            boost::algorithm::to_lower(name);
        }
        auto macs_symbol = macs::Folder::Symbol::none;
        if (symbol.size())
        {
            macs_symbol = macs::Folder::Symbol::getByTitle(symbol);
            if (macs_symbol == macs::Folder::Symbol::none)
            {
                return cb(code::symbol_not_found, folder_vector_ptr());
            }
        }
        auto parent_fid = parent.empty() ? macs::Folder::noParent : parent;
        meta_backend->mailish().createFolder(
            name,
            parent_fid,
            macs_symbol,
            factory.release(),
            [this, self = yplatform::shared_from(this), path = folder.path, cb](
                mail_errors::error_code err, macs::MailishFolder mailish_folder) {
                if (err)
                {
                    YLOG_L(error) << "can't create folder: " << mail_error_message(err);
                    return cb(err.base(), folder_vector_ptr());
                }
                folder_vector_ptr result = std::make_shared<folder_vector>();
                result->emplace_back(path, mailish_folder.fid(), mailish_folder.uidValidity());
                cb(code::ok, result);
            });
    }

    void update_folder(
        const folder& folder,
        const fid_t_opt& new_parent_fid,
        const without_data_cb& cb)
    {
        spawn<update_folder_op<MacsService>>(meta_backend, folder, new_parent_fid, cb);
    }

    void delete_mailish_folder_entry(const fid_vector& fids, const without_data_cb& cb)
    {
        meta_backend->mailish().deleteMailishFolderEntries(
            fids, [this, self = yplatform::shared_from(this), cb](mail_errors::error_code err) {
                if (err)
                {
                    YLOG_L(error) << "can't delete mailish folder entries: "
                                  << mail_error_message(err);
                    cb(err.base());
                }
                else
                {
                    cb(code::ok);
                }
            });
    }

    void delete_folder(const fid_t& fid, const without_data_cb& cb)
    {
        spawn<delete_folder_op<MacsService>>(meta_backend, fid, cb);
    }

    void clear_folder(const fid_t& fid, const without_data_cb& cb)
    {
        spawn<clear_folder_op<MacsService>>(meta_backend, fid, settings->delete_messages_chunk, cb);
    }

    void set_folder_symbol(const fid_t& fid, const std::string& symbol, const without_data_cb& cb)
    {
        spawn<set_folder_symbol_op<MacsService>>(meta_backend, fid, symbol, cb);
    }

    void set_folders_order(const fid_t& fid, const fid_t& prev_fid, const without_data_cb& cb)
    {
        macs::SortOptions sortOptions(meta_backend->folders());
        sortOptions.move(
            fid,
            prev_fid,
            [this, self = yplatform::shared_from(this), cb](
                mail_errors::error_code err, size_t /*count*/) {
                if (err)
                {
                    YLOG_L(error) << "can't set folders order: " << mail_error_message(err);
                    cb(err.base());
                }
                else
                {
                    cb(code::ok);
                }
            });
    }

    void update_flags(const fid_t&, mid_t mid, const flags_t flags, const without_data_cb& cb)
    { // TODO unused fid
        spawn<update_flags_op<MacsService>>(meta_backend, mid, flags, cb);
    }

    void change_flags(
        const mid_vector& mids,
        const flags_t& add,
        const flags_t& del,
        const without_data_cb& cb)
    {
        spawn<change_flags_op<MacsService>>(meta_backend, mids, add, del, cb);
    }

    void update_downloaded_range(
        const fid_t& fid,
        const imap_range& range,
        const without_data_cb& cb)
    {
        meta_backend->mailish().updateDownloadedRange(
            fid,
            range.bottom(),
            range.top(),
            [this, self = yplatform::shared_from(this), cb](mail_errors::error_code err) {
                if (err)
                {
                    YLOG_L(error) << "can't update downloaded range: " << mail_error_message(err);
                    return cb(err.base());
                }
                cb(code::ok);
            });
    }

    void store_message(
        uid_t uid,
        const std::string& email,
        const message& msg,
        string&& body,
        notification_type notify_type,
        const std::string& priority,
        const store_message_response_cb& cb)
    {
        if (body.size())
        {
            auto impl = std::make_shared<store_message_op<MacsService>>(
                meta_backend, context, uid, std::move(body), email, msg, notify_type, priority, cb);
            impl->logger(logger());
            yplatform::spawn(impl);
        }
        else
        {
            cb(code::body_is_empty, store_message_response());
        }
    }

    void check_spam(
        uid_t uid,
        const karma_t& karma,
        const std::string& from,
        const std::vector<std::string>& to,
        const std::string& client_ip,
        const std::string& request_id,
        string_ptr body,
        const without_data_cb& cb)
    {
        auto impl = std::make_shared<check_spam_op>(
            uid, karma, from, to, client_ip, request_id, *body, context, logger(), cb);
        (*impl)();
    }

    void increment_mailish_entry_errors_count(
        const fid_t& fid,
        imap_id_t ext_msg,
        uint32_t errors_count,
        const without_data_cb& cb)
    {
        meta_backend->mailish().incrementErrors(
            fid,
            ext_msg,
            errors_count,
            [this, self = yplatform::shared_from(this), cb](mail_errors::error_code err) {
                if (err)
                {
                    YLOG_L(error) << "can't increment mailish entry errors: "
                                  << mail_error_message(err);
                    cb(err.base());
                }
                else
                {
                    cb(code::ok);
                }
            });
    }

    void delete_mailish_entry(
        const fid_t& fid,
        const imap_id_vector& ids,
        const without_data_cb& cb)
    {
        std::vector<uint64_t> imap_ids;
        imap_ids.reserve(ids.size());
        for (auto& id : ids)
        {
            imap_ids.push_back(id);
        }
        meta_backend->mailish().deleteMailishEntries(
            fid,
            imap_ids,
            [this, self = yplatform::shared_from(this), cb](mail_errors::error_code err) {
                if (err)
                {
                    YLOG_L(error) << "can't delete mailish entries: " << mail_error_message(err);
                    cb(err.base());
                }
                else
                {
                    cb(code::ok);
                }
            });
    }

    void clear(const without_data_cb& cb)
    {
        spawn<clear_mailbox_op<MacsService>>(meta_backend, settings->delete_messages_chunk, cb);
    }

    void erase_security_locks(const without_data_cb& cb)
    {
        accounts_repository->erase_security_locks(cb);
    }

    void get_account(const account_cb& cb)
    {
        accounts_repository->get_account(cb);
    }

    void save_account(const account_t& account, const without_data_cb& cb)
    {
        accounts_repository->save_account(account, cb);
    }

    void invalidate_auth_data(const token_id_t& token, const without_data_cb& cb)
    {
        accounts_repository->invalidate_auth_data(token, cb);
    }

    void compose_draft(
        uid_t uid,
        const std::string& user_ticket,
        store_request_ptr request,
        const json_cb& cb)
    {
        sendbernar->compose_draft(uid, user_ticket, request, cb);
    }

    void compose_message(
        uid_t uid,
        const std::string& user_ticket,
        send_request_ptr request,
        const json_cb& cb)
    {
        sendbernar->compose_message(uid, user_ticket, request, cb);
    }

    void get_attachments_sids(
        uid_t uid,
        const std::string& user_ticket,
        mid_t mid,
        const std::vector<std::string>& hids,
        const sids_cb& cb)
    {
        auto impl =
            std::make_shared<get_attachments_sids_op>(context, uid, user_ticket, mid, hids, cb);
        impl->logger(logger());
        (*impl)();
    }

    void update_last_sync_ts(std::time_t last_sync_ts, const without_data_cb& cb)
    {
        accounts_repository->update_last_sync_ts(last_sync_ts, cb);
    }

    void get_karma(uid_t uid, const karma_cb& cb)
    {
        auth::get_user_info(
            context,
            { uid },
            my_ip,
            my_port,
            [this, self = yplatform::shared_from(this), uid, cb](
                error ec, const auth::user_info_response& response) {
                if (!ec)
                {
                    auto it = response.find(uid);
                    if (it != response.end())
                    {
                        return cb(ec, it->second.karma);
                    }
                    else
                    {
                        YLOG_L(error) << "user not found in get_user_info response";
                        ec = error(code::auth_error, "user not found in get_user_info response");
                    }
                }
                cb(ec, karma_t());
            });
    }

    void get_or_create_label(
        const std::string& name,
        const std::string& color,
        const std::string& type,
        bool force_create,
        const lid_cb& cb)
    {
        auto macs_cb = [self = yplatform::shared_from(this), cb](
                           error err, macs::Label label) { cb(err, err ? lid() : label.lid()); };
        if (force_create)
        {
            meta_backend->labels().createLabelWithType(name, color, type, macs_cb);
        }
        else
        {
            spawn<get_or_create_label_op<MacsService>>(meta_backend, name, color, type, macs_cb);
        }
    }

    void get_or_create_label_by_symbol(
        const std::string& symbol,
        bool force_create,
        const lid_cb& cb)
    {
        auto macs_sym = macs::Label::Symbol::getByTitle(symbol);
        if (macs_sym == macs::Label::Symbol::none)
        {
            YLOG_L(error) << "can't get or create label by symbol: unknown symbol=" << symbol;
            return cb(code::symbol_not_found, lid());
        }
        auto macs_cb = [self = yplatform::shared_from(this), cb](
                           error err, macs::Label label) { cb(err, err ? lid() : label.lid()); };
        if (force_create)
        {
            meta_backend->labels().createLabel(macs_sym, macs_cb);
        }
        else
        {
            spawn<get_or_create_label_op<MacsService>>(meta_backend, symbol, macs_cb);
        }
    }

    void update_label(
        const lid& lid,
        const std::string& name,
        const std::string& color,
        const without_data_cb& cb)
    {
        spawn<update_label_op<MacsService>>(meta_backend, lid, name, color, cb);
    }

    void delete_label(const lid& lid, const without_data_cb& cb)
    {
        meta_backend->labels().deleteLabel(
            lid,
            [this, self = yplatform::shared_from(this), cb](
                mail_errors::error_code err, macs::Revision /*rev*/) {
                if (err)
                {
                    YLOG_L(error) << "can't delete label: " << mail_error_message(err);
                    cb(err.base());
                }
                else
                {
                    cb(code::ok);
                }
            });
    }

    void get_send_operation_result(const std::string& operation_id, string_opt_cb cb)
    {
        using namespace ymod_cachedb;
        if (operation_id.empty())
        {
            return cb({}, {});
        }
        auto cachedb_client = yplatform::find<Cache>("cachedb");
        cachedb_client->get(
            Uid(uid),
            Key(operation_id),
            RequestId(context->uniq_id()),
            [this, self = yplatform::shared_from(this), cb](
                mail_errors::error_code ec, OptValue val) {
                string_opt res;
                if (ec)
                {
                    YLOG_L(error) << "can't get send operation result: " << mail_error_message(ec);
                }
                else if (val)
                {
                    res = *val;
                }
                cb(ec.base(), res);
            });
    }

    void save_send_operation_result(
        const std::string& operation_id,
        const std::string& result,
        without_data_cb cb)
    {
        using namespace ymod_cachedb;
        if (operation_id.empty())
        {
            return cb({});
        }
        auto cachedb_client = yplatform::find<Cache>("cachedb");
        cachedb_client->put(
            Uid(uid),
            Key(operation_id),
            Value(result),
            RequestId(context->uniq_id()),
            [this, self = yplatform::shared_from(this), cb](mail_errors::error_code ec, bool val) {
                if (ec || !val)
                {
                    YLOG_L(error) << "can't save send operation result: "
                                  << (ec ? mail_error_message(ec) : "put returned false");
                }
                cb(ec.base());
            });
    }

private:
    using logger_t = yplatform::log::source;

    static mailbox::message to_message(const macs::MailishMessage& mailish_message)
    {
        message message;
        message.fid = mailish_message.fid();
        message.id = static_cast<imap_id_t>(std::stoull(mailish_message.externalImapId()));
        message.mid = std::stoull(mailish_message.mid());
        message.saved_errors_count = message.errors_count = mailish_message.errors();
        message.date = mailish_message.externalReceivedDate();
        return message;
    }

    bool has_parent(const path_t& folder_path)
    {
        return !folder_path.get_parent_path().empty();
    }

    template <typename Coroutine, typename... Args>
    void spawn(Args&&... args)
    {
        auto coro = std::make_shared<Coroutine>(std::forward<Args>(args)...);
        coro->logger(logger());
        yplatform::spawn(coro);
    }

    boost::asio::io_service& io;
    context_ptr context;
    MacsService meta_backend;
    accounts_repository_ptr accounts_repository;
    sendbernar_client_ptr sendbernar;
    uid_t uid;
    local::settings_ptr settings;
};

}
