#include "sendbernar_client.h"

#include <common/http.h>

#include <ymod_httpclient/cluster_client.h>
#include <ymod_httpclient/util/url_parser.h>
#include <yplatform/encoding/url_encode.h>
#include <yplatform/util/safe_call.h>
#include <yplatform/util/sstream.h>

#include <boost/algorithm/string/join.hpp>
#include <set>

namespace xeno::mailbox::local {

namespace ph = std::placeholders;

static const std::set<std::string>& req_headers()
{
    static const std::set<std::string> headers{ "x-original-host", "x-real-ip", "x-request-id" };
    return headers;
}

static const std::set<std::string>& proxy_params()
{
    static const std::set<std::string> proxy{ "subj",      "to",         "cc",
                                              "bcc",       "from_name",  "from_mailbox",
                                              "inreplyto", "references", "parts_json" };
    return proxy;
}

sendbernar_client::sendbernar_client(context_ptr ctx, const yplatform::log::source& logger)
    : yplatform::log::contains_logger(logger), ctx(ctx)
{
}

void sendbernar_client::compose_draft(
    uid_t uid,
    const std::string& user_ticket,
    store_request_ptr request,
    const json_cb& cb)
{
    std::string url;
    yplatform::sstream url_stream(url, sizeof("/compose_draft?caller=xeno&v=1&uid="));
    url_stream << "/compose_draft?caller=xeno&v=1&uid=" << uid;

    try
    {
        auto headers = make_compose_headers(request);
        auto body = make_compose_body(request);

        auto request = http::request_t::POST(url, std::move(headers), std::move(body));
        http::append_tvm2_user_header(request, user_ticket);

        auto handler =
            std::bind(&sendbernar_client::compose_cb, shared_from_this(), cb, ph::_1, ph::_2);
        auto client = yplatform::find<yhttp::cluster_client>("sendbernar_client");
        client->async_run(ctx, std::move(request), handler);
    }
    catch (const std::exception& e)
    {
        YLOG_L(error) << "sendbernar error: exception: " << e.what();
        yplatform::safe_call(ctx, cb, code::local_mailbox_exception, Json::objectValue);
    }
}

void sendbernar_client::compose_message(
    uid_t uid,
    const std::string& user_ticket,
    send_request_ptr request,
    const json_cb& cb)
{
    std::string url;
    yplatform::sstream url_stream(url, sizeof("/compose_message?caller=xeno&v=1&uid="));
    url_stream << "/compose_message?caller=xeno&v=1&uid=" << uid;

    try
    {
        auto headers = make_compose_headers(request);
        auto body = make_compose_body(request);

        auto request = http::request_t::POST(url, std::move(headers), std::move(body));
        http::append_tvm2_user_header(request, user_ticket);

        auto handler =
            std::bind(&sendbernar_client::compose_cb, shared_from_this(), cb, ph::_1, ph::_2);
        auto client = yplatform::find<yhttp::cluster_client>("sendbernar_client");
        client->async_run(ctx, std::move(request), handler);
    }
    catch (const std::exception& e)
    {
        YLOG_L(error) << "sendbernar error: exception: " << e.what();
        yplatform::safe_call(ctx, cb, code::local_mailbox_exception, Json::objectValue);
    }
}

void sendbernar_client::compose_cb(const json_cb& cb, error ec, yhttp::response http_resp)
{
    try
    {
        if (ec)
        {
            YLOG_L(error) << "sendbernar error: " << ec.message();
            return yplatform::safe_call(ctx, cb, code::sendbernar_error, Json::objectValue);
        }

        if (http_resp.status != 200)
        {
            auto error_msg = parse_error_response(http_resp.body);
            YLOG_L(error) << "sendbernar error: bad status: " << http_resp.status << " "
                          << http_resp.reason << (error_msg.empty() ? "" : " ") << error_msg;
            ec = (http_resp.status / 100 == 4 ? code::bad_request : code::sendbernar_error);
            return yplatform::safe_call(ctx, cb, ec, Json::objectValue);
        }

        json::value result;
        json_reader reader;
        if (!reader.parse(http_resp.body, result))
        {
            YLOG_L(error) << "sendbernar error: bad json";
            ec = code::sendbernar_error;
        }
        yplatform::safe_call(ctx, cb, ec, std::move(result));
    }
    catch (const std::exception& e)
    {
        YLOG_L(error) << "sendbernar error: exception: " << e.what();
        ec = code::sendbernar_error;
        yplatform::safe_call(ctx, cb, ec, Json::objectValue);
    }
}

std::string sendbernar_client::make_compose_headers(store_request_ptr request)
{
    std::string headers_str;
    yplatform::sstream headers_stream(headers_str);
    auto& req_hdr = req_headers();
    for (auto& hdr : req_hdr)
    {
        auto value = request->get_value(hdr);
        if (value.size())
        {
            headers_stream << hdr << ": " << value << "\r\n";
        }
        else
        {
            throw std::runtime_error("cannot find require header: " + hdr);
        }
    }
    return headers_str;
}

std::string sendbernar_client::make_compose_body(store_request_ptr request)
{
    using yplatform::url_encode;
    std::vector<std::string> params;
    auto ttype = request->get_value("ttype");
    if (ttype.size())
    {
        params.push_back(std::string("html=") + (ttype == "html" ? "yes" : "no"));
    }

    auto send = request->get_value("send");
    if (send.size())
    {
        params.push_back("text=" + url_encode<std::string>(send));
        params.push_back("force7bit=yes");
    }

    bool is_source_mid_found = false;
    auto source_mid = request->get_value("draft_base");
    if (source_mid.size())
    {
        is_source_mid_found = true;
    }
    else
    {
        source_mid = request->get_value("reply");
        if (source_mid.size())
        {
            is_source_mid_found = true;
        }
        else
        {
            source_mid = request->get_value("forward");
            if (source_mid.size())
            {
                is_source_mid_found = true;
            }
        }
    }

    if (is_source_mid_found)
    {
        params.push_back("source_mid=" + url_encode<std::string>(source_mid));

        auto parts = request->get_string_array("parts");
        for (auto& hid : parts)
        {
            params.push_back(
                "parts_json=" + url_encode<std::string>(to_parts_json(source_mid, hid)));
        }
    }

    for (auto& id : request->get_string_array("ids"))
    {
        params.push_back("forward_mids=" + url_encode<std::string>(id));
    }

    for (auto& att_id : request->get_string_array("att_ids"))
    {
        params.push_back("uploaded_attach_stids=" + url_encode<std::string>(att_id));
    }

    auto narod_att = request->get_value("narod_att");
    if (narod_att.size())
    {
        params.push_back("disk_attaches=" + url_encode<std::string>(narod_att));
    }

    for (auto&& att : request->get_json_array("disk_att"))
    {
        replace_url_by_path(att, "url", "path");
        replace_url_by_path(att, "preview_url", "preview_path");
        params.push_back("disk_attaches_json=" + url_encode<std::string>(json::to_string(att)));
    }

    auto& proxy = proxy_params();
    for (auto& param : proxy)
    {
        auto value = request->get_value(param);
        if (value.size())
        {
            params.push_back(param + "=" + url_encode<std::string>(value));
        }
        else
        {
            for (auto& it : request->get_string_array(param))
            {
                params.push_back(param + "=" + url_encode<std::string>(it));
            }
        }
    }

    return boost::algorithm::join(params, "&");
}

std::string sendbernar_client::to_parts_json(const std::string& mid, const std::string& hid)
{
    json::value result;
    result["mid"] = mid;
    result["hid"] = hid;
    return json::to_string(result);
}

std::string sendbernar_client::parse_error_response(const std::string& response)
{
    json::value parsed_response;
    json_reader reader;
    if (!reader.parse(response, parsed_response))
    {
        return std::string();
    }

    auto message =
        (parsed_response["message"].isNull() ? std::string() :
                                               parsed_response["message"].asString());
    auto reason =
        (parsed_response["reason"].isNull() ? std::string() : parsed_response["reason"].asString());
    return message + (message.empty() ? "" : " ") + reason;
}

void sendbernar_client::replace_url_by_path(
    json::value& val,
    const std::string& old_key,
    const std::string& new_key)
{
    if (!val.isMember(old_key)) return;
    if (val[old_key].isNull())
    {
        val[new_key] = json::value::null;
        val.removeMember(old_key);
        return;
    }
    std::string url = val[old_key].asString();
    if (url.empty())
    {
        val[new_key] = "";
        val.removeMember(old_key);
        return;
    }
    std::string protocol, host, path;
    unsigned short port = 0;
    boost::system::error_code err = yhttp::parse_url(url, protocol, host, port, path);
    if (err)
    {
        YLOG_L(error) << "parse url error=" << err.message() << " url=" << url;
        throw std::runtime_error("cannot parse url: " + err.message());
    }
    val[new_key] = path;
    val.removeMember(old_key);
}

}
