#pragma once

#include <common/mail_errors.h>

#include <macs_pg/macs_pg.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

namespace xeno::mailbox::local {

template <typename MacsService>
class set_folder_symbol_op : public yplatform::log::contains_logger
{
public:
    using yield_context = yplatform::yield_context<set_folder_symbol_op>;

    set_folder_symbol_op(
        MacsService service,
        const fid_t& fid,
        const std::string& symbol,
        const without_data_cb& cb)
        : service_(service), fid_(fid), symbol_(symbol), cb_(cb)
    {
    }

    void operator()(yield_context ctx, error ec = {})
    {
        try
        {
            reenter(ctx)
            {
                yield service_->folders().getAllFolders(ctx);
                if (ec) yield break;
                if (!folders_.exists(fid_))
                {
                    ec = code::folder_not_found;
                    yield break;
                }
                if (symbol_.empty())
                {
                    yield service_->folders().resetSymbol(folders_.at(fid_), ctx);
                }
                else
                {
                    yield set_folder_symbol(ctx);
                }
            }
        }
        catch (const std::exception& e)
        {
            YLOG_L(error) << "set folder symbol op exception: " << e.what();
            ec = code::local_mailbox_exception;
        }

        if (ctx.is_complete())
        {
            cb_(ec);
        }
    }

    void operator()(yield_context ctx, mail_errors::error_code err, macs::FolderSet folders)
    {
        if (!err)
        {
            folders_ = std::move(folders);
        }
        else
        {
            YLOG_L(error) << "set folder symbol op error: " << mail_error_message(err);
        }
        (*this)(ctx, err.base());
    }

    void operator()(yield_context ctx, mail_errors::error_code err, macs::Revision /*rev*/)
    {
        if (err)
        {
            YLOG_L(error) << "set folder symbol op error: " << mail_error_message(err);
        }
        (*this)(ctx, err.base());
    }

private:
    void set_folder_symbol(yield_context ctx)
    {
        auto macs_symbol = macs::Folder::Symbol::getByTitle(symbol_);
        if (macs_symbol == macs::Folder::Symbol::none)
        {
            return (*this)(ctx, code::symbol_not_found);
        }
        service_->folders().setSymbol(folders_.at(fid_), macs_symbol, ctx);
    }

    MacsService service_;
    fid_t fid_;
    std::string symbol_;
    without_data_cb cb_;
    macs::FolderSet folders_;
};

}

#include <yplatform/unyield.h>
