#pragma once

#include "get_or_create_labels_op.h"

#include <common/context.h>
#include <common/http.h>
#include <common/json.h>

#include <ymod_httpclient/cluster_client.h>
#include <yplatform/coroutine.h>
#include <yplatform/find.h>
#include <boost/lexical_cast.hpp>

#include <yplatform/yield.h>
namespace xeno::mailbox::local {
template <typename MacsService>
struct store_message_op
    : public std::enable_shared_from_this<store_message_op<MacsService>>
    , public yplatform::log::contains_logger
{
    using yield_context = yplatform::yield_context<store_message_op>;

    store_message_op(
        MacsService service,
        const context_ptr& context,
        uid_t uid,
        string&& body,
        const std::string& email,
        const message& msg,
        notification_type notify_type,
        const std::string& priority,
        const store_message_response_cb& handler)
        : service(service)
        , context(context)
        , uid(uid)
        , body(std::move(body))
        , email(email)
        , msg(msg)
        , notify_type(notify_type)
        , priority(priority)
        , handler(handler)
    {
    }

    void operator()(yield_context ctx)
    {
        try
        {
            reenter(ctx)
            {
                yield convert_flags_to_labels(ctx.capture(ec, labels), msg.flags);
                if (ec) yield break;

                yield store_message(ctx.capture(ec, http_resp));
                if (ec) yield break;

                if (http_resp.status == 409)
                {
                    ec = code::message_is_duplicate;
                    yield break;
                }

                if (http_resp.status != 200)
                {
                    YLOG_L(error) << "store message error, bad status: " << http_resp.status << " "
                                  << http_resp.reason;
                    ec = error(code::store_message_error, http_resp.body);
                    yield break;
                }

                store_result = parse_response(http_resp);
            }
        }
        catch (const std::exception& e)
        {
            YLOG_L(error) << "store message op exception: " << e.what();
            ec = code::local_mailbox_exception;
        }

        if (ctx.is_complete())
        {
            handler(ec, store_result);
        }
    }

    template <typename Handler>
    void convert_flags_to_labels(Handler&& handler, const flags_t& flags)
    {
        auto coro = std::make_shared<get_or_create_labels_op<MacsService>>(
            service, flags, std::move(handler));
        coro->logger(logger());
        yplatform::spawn(coro);
    }

    template <typename Handler>
    void store_message(Handler&& handler)
    {
        auto client = yplatform::find<yhttp::cluster_client>("nw_client");
        auto params = yhttp::url_encode({ { "service", "xeno" },
                                          { "uid", uid },
                                          { "fid", msg.fid },
                                          { "external_imap_id", msg.id },
                                          { "priority", priority } });
        auto headers = "X-Request-Id: " + context->uniq_id() + "\r\n";
        auto multipart = yhttp::body_mixed(
            { yhttp::json_part(make_json_args()), yhttp::rfc822_part(std::move(body)) });
        auto req = yhttp::request::MPOST(
            "/mail/store_mailish" + params, std::move(headers), std::move(multipart));
        client->async_run(context, std::move(req), handler);
    }

    std::string make_json_args()
    {
        json::value res;
        json::value options;
        options["enable_push"] = notify_type != notification_type::disabled;
        res["options"] = options;

        json::value mail_info;
        mail_info["received_date"] = msg.date;
        mail_info["labels"] = make_labels_arg();
        res["mail_info"] = mail_info;

        json::value user_info;
        user_info["email"] = email;
        res["user_info"] = user_info;

        return json::to_string(res);
    }

    json::value make_labels_arg()
    {
        json::value symbols(Json::arrayValue);
        json::value lids(Json::arrayValue);
        for (const auto& label : labels)
        {
            if (label.symbolicName().title().empty())
            {
                lids.append(label.lid());
            }
            else
            {
                symbols.append(label.symbolicName().title());
            }
        }

        json::value labels;
        labels["symbol"] = symbols;
        labels["lids"] = lids;
        labels["imap"] = Json::arrayValue;
        labels["system"] = Json::arrayValue;
        return labels;
    }

    store_message_response parse_response(const yhttp::response& http_resp) const
    {
        yplatform::json_value response;
        if (auto error = response.parse(http_resp.body))
        {
            throw std::runtime_error("json parse error: " + *error);
        }

        store_message_response res;
        res.mid = boost::lexical_cast<mid_t>(response["mid"].to_string());
        res.int_imap_id = boost::lexical_cast<imap_id_t>(response["imap_id"].to_string());
        return res;
    }

    MacsService service;
    context_ptr context;
    uid_t uid;
    string body;
    std::string email;
    message msg;
    notification_type notify_type;
    std::string priority;
    store_message_response_cb handler;

    error ec;
    std::list<macs::Label> labels;
    yhttp::response http_resp;
    store_message_response store_result;
};

}

#include <yplatform/unyield.h>
