#pragma once

#include <common/mail_errors.h>
#include <mailbox/local/get_or_create_labels_op.h>

#include <macs_pg/macs_pg.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

namespace xeno::mailbox::local {

template <typename MacsService>
class update_flags_op : public yplatform::log::contains_logger
{
public:
    using yield_context = yplatform::yield_context<update_flags_op>;
    using labels_t = std::list<macs::Label>;

    update_flags_op(
        MacsService service,
        const mid_t mid,
        const flags_t& flags,
        const without_data_cb& cb)
        : service_(service), mid_(std::to_string(mid)), flags_(flags), cb_(cb)
    {
    }

    void operator()(yield_context ctx)
    {
        labels_t add_labels, del_labels;
        std::string flag;
        reenter(ctx)
        {
            yield service_->labels().getAllLabels(ctx);
            yield convert_flags_to_labels(ctx);
            yield get_current_labels(ctx);

            yield mark_labels(ctx);
            yield unmark_labels(ctx);

            cb_(code::ok);
        }
    }

    template <typename... Args>
    void operator()(yield_context ctx, mail_errors::error_code err, Args&&... args)
    {
        if (err)
        {
            YLOG_L(error) << "can't update flags: " << mail_error_message(err);
            cb_(err.base());
        }
        else
        {
            (*this)(ctx, std::forward<Args>(args)...);
        }
    }

    void operator()(yield_context ctx, error err, labels_t labels)
    {
        if (err)
        {
            cb_(err);
        }
        else
        {
            new_labels_ = std::move(labels);
            (*this)(ctx);
        }
    }

    void operator()(yield_context ctx, macs::LabelSet labels)
    {
        all_labels_ = std::move(labels);
        (*this)(ctx);
    }

    void operator()(yield_context ctx, macs::Envelope envelope)
    {
        // extract current labels from envelope
        for (auto& lid : envelope.labels())
        {
            auto it = all_labels_.find(lid);
            if (it == all_labels_.end())
            {
                return cb_(code::label_not_found);
            }
            if (is_syncable_label(it->second))
            {
                current_labels_.emplace(lid, it->second);
            }
        }
        (*this)(ctx);
    }

    void operator()(yield_context ctx, macs::Revision)
    {
        (*this)(ctx);
    }

private:
    using lids_t = std::vector<macs::Lid>;
    using system_flag_t = ::xeno::mailbox::system_flag_t;

    void convert_flags_to_labels(yield_context ctx)
    {
        auto coro = std::make_shared<get_or_create_labels_op<MacsService>>(service_, flags_, ctx);
        coro->logger(logger());
        yplatform::spawn(coro);
    }

    void get_current_labels(yield_context ctx)
    {
        service_->envelopes().getById(mid_, ctx);
    }

    void mark_labels(yield_context ctx)
    {
        auto add_labels = get_add_labels();
        if (!add_labels.empty())
        {
            service_->envelopes().markEnvelopes(macs::MidList{ mid_ }, add_labels, ctx);
        }
        else
        {
            ctx();
        }
    }

    void unmark_labels(yield_context ctx)
    {
        auto del_labels = get_del_labels();
        if (!del_labels.empty())
        {
            service_->envelopes().unmarkEnvelopes(macs::MidList{ mid_ }, del_labels, ctx);
        }
        else
        {
            ctx();
        }
    }

    bool is_syncable_label(const macs::Label& label)
    {
        return label.isUser() || label.symbolicName() == macs::Label::Symbol::seen_label ||
            label.symbolicName() == macs::Label::Symbol::deleted_label ||
            label.symbolicName() == macs::Label::Symbol::recent_label ||
            label.symbolicName() == macs::Label::Symbol::answered_label ||
            label.symbolicName() == macs::Label::Symbol::important_label ||
            label.symbolicName() == macs::Label::Symbol::draft_label;
    }

    labels_t get_add_labels()
    {
        labels_t ret;
        for (auto& label : new_labels_)
        {
            if (!current_labels_.exists(label.lid()))
            {
                ret.push_back(label);
            }
        }
        return ret;
    }

    labels_t get_del_labels()
    {
        labels_t ret;
        for (auto& [lid, label] : current_labels_)
        {
            auto it = std::find_if(
                new_labels_.begin(), new_labels_.end(), [lid = lid](const macs::Label& label) {
                    return label.lid() == lid;
                });
            if (it == new_labels_.end())
            {
                ret.push_back(label);
            }
        }
        return ret;
    }

    MacsService service_;
    macs::Mid mid_;
    flags_t flags_;
    without_data_cb cb_;
    macs::LabelSet all_labels_;
    macs::LabelSet current_labels_;
    labels_t new_labels_;
};

}

#include <yplatform/unyield.h>
