#pragma once

#include <common/mail_errors.h>
#include <mailbox/data_types/folder.h>
#include <common/errors.h>

#include <macs_pg/macs_pg.h>
#include <macs_pg/mailish/folder_info_factory.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

namespace xeno::mailbox::local {

template <typename MacsService>
class update_folder_op : public yplatform::log::contains_logger
{
public:
    using yield_context = yplatform::yield_context<update_folder_op>;

    update_folder_op(
        MacsService service,
        const folder& folder,
        const fid_t_opt& new_parent_fid,
        const without_data_cb& cb)
        : service_(service), folder_(folder), new_parent_fid_(new_parent_fid), cb_(cb)
    {
    }

    void operator()(yield_context ctx, mail_errors::error_code err = mail_errors::error_code())
    {
        if (err)
        {
            YLOG_L(error) << "can't update folder: " << mail_error_message(err);
            return cb_(err.base());
        }
        reenter(ctx)
        {
            yield service_->folders().getAllFolders(ctx);

            if (!folders_.exists(folder_.fid))
            {
                return cb_(code::folder_not_found);
            }

            if (new_parent_fid_ && *new_parent_fid_ != "0" && !folders_.exists(*new_parent_fid_))
            {
                return cb_(code::parent_not_found);
            }

            yield update_mailish_folder(ctx);

            cb_(code::ok);
        }
    }

    void operator()(yield_context ctx, mail_errors::error_code err, macs::FolderSet folders)
    {
        folders_ = std::move(folders);
        (*this)(ctx, err);
    }

private:
    macs::Folder get_new_folder()
    {
        auto original_folder = folders_.at(folder_.fid);
        if (original_folder.isSystem()) return original_folder;
        macs::FolderFactory factory(original_folder);
        factory.name(folder_.path.get_name());
        if (new_parent_fid_)
        {
            factory.parentId(*new_parent_fid_);
        }
        return factory.product();
    }

    void update_mailish_folder(yield_context ctx)
    {
        auto folder = get_new_folder();
        macs::MailishFolderInfoFactory factory;
        factory.externalPath(folder_.path.to_string()).uidValidity(folder_.uidvalidity);
        service_->mailish().updateFolder(folder, factory.release(), ctx);
    }

    MacsService service_;
    folder folder_;
    fid_t_opt new_parent_fid_;
    without_data_cb cb_;
    macs::FolderSet folders_;
};

}

#include <yplatform/unyield.h>
