#pragma once

#include <common/mail_errors.h>

#include <macs_pg/macs_pg.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

namespace xeno::mailbox::local {

template <typename MacsService>
class update_label_op : public yplatform::log::contains_logger
{
public:
    using yield_context = yplatform::yield_context<update_label_op>;

    update_label_op(
        MacsService service,
        const lid& lid,
        const std::string& name,
        const std::string& color,
        const without_data_cb& cb)
        : service_(service), lid_(lid), name_(name), color_(color), cb_(cb)
    {
    }

    void operator()(yield_context ctx, mail_errors::error_code err = mail_errors::error_code())
    {
        if (err)
        {
            YLOG_L(error) << "can't update label: " << mail_error_message(err);
            return cb_(err.base());
        }
        reenter(ctx)
        {
            yield service_->labels().getAllLabels(ctx);
            yield update_label(ctx);
            cb_(code::ok);
        }
    }

    void operator()(yield_context ctx, mail_errors::error_code err, macs::LabelSet labels)
    {
        all_labels_ = std::move(labels);
        (*this)(ctx, err);
    }

    void operator()(yield_context ctx, mail_errors::error_code err, macs::Label /*label*/)
    {
        (*this)(ctx, err);
    }

private:
    void update_label(yield_context ctx)
    {
        auto it = all_labels_.find(lid_);
        if (it == all_labels_.end())
        {
            return cb_(code::label_not_found);
        }
        auto factory = macs::LabelFactory(it->second);
        if (color_.size())
        {
            factory.color(color_);
        }
        if (name_.size())
        {
            factory.name(name_);
        }
        service_->labels().updateLabel(factory.product(), ctx);
    }

    MacsService service_;
    lid lid_;
    std::string name_;
    std::string color_;
    without_data_cb cb_;
    macs::LabelSet all_labels_;
};

}

#include <yplatform/unyield.h>
