#pragma once

#include "uid_resolver.h"
#include <encryption/module.h>
#include <macs_pg/service/service.h>
#include <macs_pg/mailish/account_factory.h>
#include <macs_pg/mailish/auth_data_factory.h>
#include <mailbox/common.h>

namespace xeno::mdb {

class accounts_repository : public std::enable_shared_from_this<accounts_repository>
{
public:
    using account_cb = mailbox::account_cb;

    accounts_repository(context_ptr context, uid_t uid, const shard_id& shard_id, bool use_replica)
        : uid(uid), encryption(yplatform::find<encryption::module>("encryption"))
    {
        auto factory = std::make_shared<uid_resolver_factory>(shard_id);
        auto macs = yplatform::find<ymod_macs::module>("macs");
        service = macs->get_service(context, std::to_string(uid), use_replica, factory);
    }

    void erase_security_locks(const without_data_cb& cb)
    {
        service->mailish().eraseSecurityLocks(
            [self = shared_from_this(), cb](mail_errors::error_code err) {
                if (err)
                {
                    YLOG_GLOBAL(error) << "can't erase security locks: " << err.what();
                    cb(err.base());
                    return;
                }
                cb(code::ok);
            });
    }

    void get_account(const account_cb& cb)
    {
        service->mailish().getAccount(
            [this, self = shared_from_this(), cb](
                mail_errors::error_code err, macs::MailishAccount mailish_account) {
                if (err)
                {
                    YLOG_GLOBAL(error) << "can't get account: " << err.what();
                    cb(err.base(), account_t());
                    return;
                }
                service->mailish().getAuthData(
                    [this, self, mailish_account, cb](
                        mail_errors::error_code err, macs::MailishAuthDataChunk chunk) {
                        try
                        {
                            if (err)
                            {
                                throw std::runtime_error(err.what());
                            }
                            auto account = to_account(mailish_account, chunk);
                            cb(code::ok, std::move(account));
                        }
                        catch (const std::exception& e)
                        {
                            YLOG_GLOBAL(error) << "can't get auth data: " << e.what();
                            cb(code::local_mailbox_exception, account_t());
                        }
                    });
            });
    }

    void save_account(const account_t& account, const without_data_cb& cb)
    {
        if (account.auth_data.size() != 1)
        {
            YLOG_GLOBAL(error)
                << "can't save account: account should have exactly one auth_data but given "
                << account.auth_data.size();
            return cb(code::cannot_save_account);
        }

        macs::MailishAccount mailish_account = to_mailish_account(account);
        macs::MailishAuthData mailish_auth_data = to_mailish_auth_data(account);

        service->mailish().saveAccount(
            mailish_auth_data,
            mailish_account,
            [self = shared_from_this(), cb](mail_errors::error_code err) {
                if (err)
                {
                    YLOG_GLOBAL(error) << "can't save account: " << err.what();
                    cb(err.base());
                }
                else
                {
                    cb(code::ok);
                }
            });
    }

    void invalidate_auth_data(const token_id_t& token, const without_data_cb& cb)
    {
        // TODO invalidate passport token by token_id
        // TODO unsubscribe xiva by uuid(not in the interface now, i think  const auth_data& needed
        // instead of token)
        service->mailish().invalidateAuthData(token, [cb](mail_errors::error_code err) {
            if (err)
            {
                YLOG_GLOBAL(error) << "can't invalidate auth data: " << err.what();
                cb(err.base());
            }
            else
            {
                cb(code::ok);
            }
        });
    }

    void update_last_sync_ts(std::time_t last_sync_ts, const without_data_cb& cb)
    {
        service->mailish().updateLastSyncTs(
            last_sync_ts, [self = shared_from_this(), cb](mail_errors::error_code err) {
                error ec;
                if (err)
                {
                    YLOG_GLOBAL(error) << "can't update last sync: " << err.what();
                    ec = err.base();
                }
                cb(ec);
            });
    }

private:
    account_t to_account(
        const macs::MailishAccount& account,
        const macs::MailishAuthDataChunk& mailish_auth_data_chunk)
    {
        account_t ret;
        ret.uid = uid;
        ret.email = account.email();
        ret.imap_login = account.imapLogin().size() ? account.imapLogin() : account.email();
        ret.imap_ep = endpoint(
            account.imapServer(), static_cast<uint16_t>(account.imapPort()), account.imapSsl());
        ret.smtp_login = account.smtpLogin().size() ? account.smtpLogin() : account.email();
        ret.smtp_ep = endpoint(
            account.smtpServer(), static_cast<uint16_t>(account.smtpPort()), account.smtpSsl());
        ret.last_sync_ts = account.lastSyncTs();
        for (auto& mailish_auth_data : mailish_auth_data_chunk)
        {
            try
            {
                ret.auth_data.push_back(to_auth_data(account.email(), mailish_auth_data));
            }
            catch (const std::exception& e)
            {
                YLOG_GLOBAL(error)
                    << "add auth data error: " << e.what() << ", email=" << account.email();
            }
        }
        return ret;
    }

    macs::MailishAccount to_mailish_account(const account_t& account)
    {
        macs::MailishAccountFactory account_factory;
        account_factory.email(account.email)
            .imapLogin(account.imap_login)
            .imapServer(account.imap_ep.host)
            .imapPort(account.imap_ep.port)
            .imapSsl(account.imap_ep.ssl)
            .smtpLogin(account.smtp_login)
            .smtpServer(account.smtp_ep.host)
            .smtpPort(account.smtp_ep.port)
            .smtpSsl(account.smtp_ep.ssl)
            .lastSyncTs(account.last_sync_ts);
        return account_factory.release();
    }

    auth_data to_auth_data(const std::string& email, const macs::MailishAuthData& mailish_auth_data)
    {
        auth_data auth_data;
        auth_data.xtoken_id = mailish_auth_data.tokenId();
        auth_data.uuid = mailish_auth_data.uuid();
        auth_data.type = from_macs_auth_type(mailish_auth_data.authType());
        auth_data.imap_credentials =
            encryption->decrypt(mailish_auth_data.imapCredentials(), email);
        auth_data.smtp_credentials =
            encryption->decrypt(mailish_auth_data.smtpCredentials(), email);
        auth_data.oauth_app = mailish_auth_data.oauthApplication();
        auth_data.last_valid = mailish_auth_data.lastValid();
        auth_data.security_lock = mailish_auth_data.lockFlag();
        return auth_data;
    }

    macs::MailishAuthData to_mailish_auth_data(const account_t& account)
    {
        auto auth_data = account.auth_data.begin();
        macs::MailishAuthDataFactory auth_data_factory;
        auto type = to_macs_auth_type(auth_data->type);
        auto imap_credentials = encryption->encrypt(auth_data->imap_credentials, account.email);
        auto smtp_credentials = encryption->encrypt(auth_data->smtp_credentials, account.email);

        auth_data_factory.tokenId(auth_data->xtoken_id)
            .authType(type)
            .imapCredentials(imap_credentials)
            .smtpCredentials(smtp_credentials)
            .uuid(auth_data->uuid)
            .oauthApplication(auth_data->oauth_app)
            .lastValid(auth_data->last_valid)
            .lockFlag(auth_data->security_lock);
        return auth_data_factory.release();
    }

    auth_type from_macs_auth_type(const macs::MailishAuthData::AuthType& type)
    {
        switch (type)
        {
        case macs::MailishAuthData::AuthType::OAuth2:
            return auth_type::oauth;
        case macs::MailishAuthData::AuthType::Password:
            return auth_type::password;
        default:
            throw std::runtime_error("unknown MailishAuthData::AuthType");
        }
    }

    macs::MailishAuthData::AuthType to_macs_auth_type(const auth_type& type)
    {
        switch (type)
        {
        case auth_type::oauth:
            return macs::MailishAuthData::AuthType::OAuth2;
        case auth_type::password:
            return macs::MailishAuthData::AuthType::Password;
        default:
            throw std::runtime_error("unknown auth_type");
        }
    }

    macs::ServicePtr service;
    uid_t uid;
    boost::shared_ptr<encryption::module> encryption;
};

using accounts_repository_ptr = std::shared_ptr<accounts_repository>;

}
