#pragma once

#include <ymod_macs/module.h>
#include <macs_pg/service/service.h>
#include <yplatform/find.h>
#include <utility>

namespace xeno::mdb {

struct EnvelopesRepository
{
    EnvelopesRepository(
        macs::ServicePtr master_only_service,
        macs::ServicePtr replica_than_master_service)
        : master_only_service(master_only_service)
        , replica_than_master_service(replica_than_master_service)
    {
    }

    template <typename... Args>
    void getById(Args&&... args) const
    {
        master_only_service->envelopes().getById(std::forward<Args>(args)...);
    }

    template <typename... Args>
    void getByIds(Args&&... args) const
    {
        replica_than_master_service->envelopes().getByIds(std::forward<Args>(args)...);
    }

    template <typename... Args>
    void markEnvelopes(Args&&... args) const
    {
        master_only_service->envelopes().markEnvelopes(std::forward<Args>(args)...);
    }

    template <typename... Args>
    void unmarkEnvelopes(Args&&... args) const
    {
        master_only_service->envelopes().unmarkEnvelopes(std::forward<Args>(args)...);
    }

    template <typename... Args>
    void remove(Args&&... args) const
    {
        master_only_service->envelopes().remove(std::forward<Args>(args)...);
    }

    template <typename... Args>
    void getMidsByFolder(Args&&... args) const
    {
        master_only_service->envelopes().getMidsByFolder(std::forward<Args>(args)...);
    }

    macs::ServicePtr master_only_service;
    macs::ServicePtr replica_than_master_service;
};

struct LabelsRepository : public std::enable_shared_from_this<LabelsRepository>
{
    LabelsRepository(
        macs::ServicePtr master_only_service,
        macs::ServicePtr replica_than_master_service)
        : master_only_service(master_only_service)
        , replica_than_master_service(replica_than_master_service)
    {
    }

    template <typename... Args>
    void getAllLabels(Args&&... args) const
    {
        master_only_service->labels().getAllLabels(std::forward<Args>(args)...);
    }

    template <typename Handler>
    void createLabelWithType(
        const std::string& name,
        const std::string& color,
        const std::string& type,
        Handler&& handler) const
    {
        master_only_service->labels().createLabelWithType(
            name, color, type, wrapHandler(std::forward<Handler>(handler)));
    }

    template <typename Handler>
    void createLabel(const macs::Label::Symbol& symbol, Handler&& handler) const
    {
        master_only_service->labels().createLabel(
            symbol, wrapHandler(std::forward<Handler>(handler)));
    }

    template <typename Handler>
    void deleteLabel(const std::string& lid, Handler&& handler) const
    {
        master_only_service->labels().deleteLabel(lid, wrapHandler(std::forward<Handler>(handler)));
    }

    template <typename Handler>
    void updateLabel(macs::Label label, Handler&& handler) const
    {
        master_only_service->labels().updateLabel(
            label, wrapHandler(std::forward<Handler>(handler)));
    }

    void resetLabelsCache() const
    {
        master_only_service->labels().resetLabelsCache();
        replica_than_master_service->labels().resetLabelsCache();
    }

    template <typename Handler>
    auto wrapHandler(Handler&& handler) const
    {
        return [this, self = yplatform::shared_from(this), handler](auto&& err, auto&& data) {
            resetLabelsCache();
            handler(std::move(err), std::move(data));
        };
    }

    macs::ServicePtr master_only_service;
    macs::ServicePtr replica_than_master_service;
};

struct Service
{
    Service(yplatform::task_context_ptr ctx, uid_t uid, const std::string& shard_id)
    {
        auto macs = yplatform::find<ymod_macs::module>("macs");
        auto uid_resolver_factory = std::make_shared<mdb::uid_resolver_factory>(shard_id);
        master_only_service =
            macs->get_service(ctx, std::to_string(uid), false, uid_resolver_factory);
        replica_than_master_service =
            macs->get_service(ctx, std::to_string(uid), true, uid_resolver_factory);
        envelopes_repo =
            std::make_shared<EnvelopesRepository>(master_only_service, replica_than_master_service);
        labels_repo =
            std::make_shared<LabelsRepository>(master_only_service, replica_than_master_service);
    }

    const EnvelopesRepository& envelopes() const
    {
        return *envelopes_repo;
    }

    const macs::FoldersRepository& folders() const
    {
        return master_only_service->folders();
    }

    const LabelsRepository& labels() const
    {
        return *labels_repo;
    }

    const macs::ThreadsRepository& threads() const
    {
        return master_only_service->threads();
    }

    const macs::MailishRepository& mailish() const
    {
        return master_only_service->mailish();
    }

    macs::ServicePtr master_only_service;
    macs::ServicePtr replica_than_master_service;
    std::shared_ptr<EnvelopesRepository> envelopes_repo;
    std::shared_ptr<LabelsRepository> labels_repo;
};

using ServicePtr = std::shared_ptr<Service>;

}